<?php

namespace Modules\Amazon\Http\Resources;

use App\DataTable\NewDataTableResource;
use Illuminate\Http\Resources\MissingValue;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonLedgerSummaryAdjustmentWithTypeData;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonLedgerSummaryAdjustment;
use Modules\Amazon\Enums\Entities\AmazonLedgerSummaryAdjustmentTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Repositories\AmazonLedgerSummaryRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;

/**
 * @mixin AmazonFbaReportInventoryLedgerSummary
 */
class AmazonFbaReportInventoryLedgerSummaryResource extends NewDataTableResource
{
    public function getCustomColumns(): array
    {
        $adjustments = $this->getAdjustments();
        return [
            'fnsku_product' => $this->getFnskuProduct(),
            'product' => $this->getProduct(),
            'inventory_adjustments' => $adjustments,
            'ledger_summaries' => $this->getLedgerSummaries($adjustments),
        ];
    }

    private function getAdjustments()
    {
        return $this->whenLoaded('ledgerSummaryAdjustments', fn() => $this->ledgerSummaryAdjustments->map(fn(AmazonLedgerSummaryAdjustment $adjustment) => AmazonLedgerSummaryAdjustmentWithTypeData::from([
            'id' => $adjustment->id,
            'type' => $adjustment->type->value,
            'inventory_adjustment_id' => $adjustment->inventoryAdjustment->id,
            'quantity' => $adjustment->inventoryAdjustment->getQuantity(),
        ])));
    }

    private function getFnskuProduct()
    {
        return $this->inclusive('fnsku_product', $this->whenLoaded(
            'amazonFnskuProduct',
            $this->amazonFnskuProduct?->only(['id', 'reconciled_quantity'])
        ));
    }

    private function getProduct()
    {
        return $this->when(
            $this->amazonFnskuProduct && $this->amazonFnskuProduct->relationLoaded('product'),
            function () {
                return [
                    'id' => $this->amazonFnskuProduct->product->id,
                    'sku' => $this->amazonFnskuProduct->product->sku,
                ];
            }
        );
    }

    private function getLedgerSummaries($adjustments): MissingValue|Collection
    {
        return $this->whenLoaded('ledgers', function () use ($adjustments) {
            $summaries = $this->ledgers->groupBy('event_type')->map(function ($ledgers) {
                return [
                    'total' => $ledgers->sum('quantity'),
                    'reconciled' => $ledgers->whereNotNull('reconciled_at')->sum('quantity'),
                ];
            });

            $unknownEventAdjustment = $adjustments->where('type', AmazonLedgerSummaryAdjustmentTypeEnum::UnknownEventsAdjustment)->first() ?? null;
            $totalInventoryAdjustment = $adjustments->where('type', AmazonLedgerSummaryAdjustmentTypeEnum::TotalInventoryAdjustment)->first() ?? null;
            $missingWithoutSummaryReportAdjustment = $adjustments->where('type', AmazonLedgerSummaryAdjustmentTypeEnum::MissingWithoutSummaryReportAdjustment)->first() ?? null;

            // Total Inventory Adjustment Handler

            $previousLedgerSummary = app(AmazonLedgerSummaryRepository::class)->getPreviousSummary($this->resource);

            $calculatedTotalInventoryQuantity =
                ($previousLedgerSummary->total_inventory_quantity ?? 0) +
                $this->ledgers->sum('quantity') +
                $this->unknown_events +
                ($totalInventoryAdjustment?->quantity ?? 0);

            // Missing Without Summary Report Adjustment Handler

            $missingWithoutSummaryReport =
                !app(AmazonLedgerSummaryRepository::class)->getNextSummary($this->resource) &&
                app(AmazonReportRepository::class)->wasReportTypeProcessedForDate(
                    AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
                    $this->event_datetime->addDay()) &&
                $this->total_inventory_quantity > 0;

            $totals = collect([
                AmazonLedgerSummaryAdjustmentTypeEnum::UnknownEventsAdjustment->value => [
                    'total' => $unknownEventAdjustment?->quantity ?? 0,
                    'reconciled' => $unknownEventAdjustment?->quantity ?? 0,
                ],
                AmazonLedgerSummaryAdjustmentTypeEnum::TotalInventoryAdjustment->value => [
                    'previousTotal' => $previousLedgerSummary->total_inventory_quantity ?? 0,
                    'ledgerQuantity' => $this->ledgers->sum('quantity'),
                    'adjustmentFromUnknownEventsQuantity' => $this->unknown_events,
                    'adjustmentFromSummaryQuantity' => $totalInventoryAdjustment?->quantity ?? 0,
                    'calculatedTotal' => $calculatedTotalInventoryQuantity,
                    'discrepancy' => $this->total_inventory_quantity - $calculatedTotalInventoryQuantity,
                ],
                AmazonLedgerSummaryAdjustmentTypeEnum::MissingWithoutSummaryReportAdjustment->value => [
                    'discrepancy' => $missingWithoutSummaryReport ? -$this->total_inventory_quantity : 0,
                    'adjustmentFromMissingQuantity' => $missingWithoutSummaryReportAdjustment?->quantity ?? 0,
                ],
            ]);

            return $summaries->isEmpty() ? $totals : $summaries->merge($totals);
        });
    }
}