<?php

namespace Modules\Amazon\Http\Controllers;

use App\DataTable\DataTableResource;
use App\Exceptions\StoredFileDoesNotExistException;
use App\Exports\AmazonReportSettlementTypeMappingsExport;
use App\Response;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Exceptions\AmazonAccountingMappingFileEmptyException;
use Modules\Amazon\Http\Resources\AmazonReportSettlementDataResource;
use Modules\Amazon\Http\Resources\AmazonReportSettlementTypeMappingResource;
use Modules\Amazon\Managers\AmazonReportSettlementDataManager;
use Modules\Amazon\Repositories\AmazonFinanceRepository;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class AmazonReportSettlementDataController extends AbstractAmazonDataTableController
{
    public function __construct(
        private AmazonFinanceRepository $amazonFinanceRepository,
        private AmazonReportSettlementDataManager $amazonReportSettlementDataManager
    )
    {
        parent::__construct();
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonReportSettlementData $amazonReportSettlementData): ?Response
    {
        return $this->response->addData($this->getResource()::make($amazonReportSettlementData));
    }

    public function getTypeMappings(Request $request, AmazonIntegrationInstance $integrationInstance)
    {

        $mappings = $this->amazonFinanceRepository->getTypeMappingsForIntegrationInstance(
            $integrationInstance
        );

        $mappings->load('nominalCode');

        return AmazonReportSettlementTypeMappingResource::collection($mappings);
    }

    public function mapSettlementTypes(Request $request)
    {
        $request->validate([
            'mappings' => 'required|array|min:1',
            'mappings.*.id' => [
                'required',
                Rule::exists('amazon_report_settlement_type_mappings', 'id'),
            ],
            'mappings.*.nominal_code_id' => [
                'required',
                Rule::exists('nominal_codes', 'id'),
            ],
        ]);

        $this->amazonFinanceRepository->mapSettlementTypes(
            $request->get('mappings')
        );

        return $this->response->setMessage(__('messages.success.create', ['resource' => 'settlement type mappings']));
    }

    public function exportSettlementTypes(AmazonIntegrationInstance $integrationInstance): BinaryFileResponse
    {
        $mappings = $this->amazonFinanceRepository->getTypeMappingsForIntegrationInstance(
            $integrationInstance
        );

        $mappings->load('nominalCode');

        return Excel::download(new AmazonReportSettlementTypeMappingsExport($mappings),
            'settlement-type-mappings-'.now()->format('Y-m-d_his').'.csv');
    }

    public function importSettlementTypes(AmazonIntegrationInstance $integrationInstance, Request $request): Response
    {
        $request->validate(['stored_name' => 'required']);

        try {
            $this->amazonReportSettlementDataManager->importSettlementTypes($integrationInstance->id,
                $request->get('stored_name'));
        } catch (StoredFileDoesNotExistException) {
            return $this->response->addError(
                'File not exists', Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        } catch (AmazonAccountingMappingFileEmptyException) {
            return $this->response->addError(
                'The file is empty', Response::HTTP_BAD_REQUEST, ''
            )->error(Response::HTTP_BAD_REQUEST);
        }

        $mappings = $this->amazonFinanceRepository->getTypeMappingsForIntegrationInstance(
            $integrationInstance
        );

        $mappings->load('nominalCode');

        return $this->response->addData(AmazonReportSettlementTypeMappingResource::collection($mappings));
    }

    protected function getModel(): string
    {
        return AmazonReportSettlementData::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonReportSettlementDataResource::class;
    }
}
