<?php

namespace Modules\Amazon\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Response;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Http\Request;
use Modules\Amazon\Actions\CreateAmazonReport;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Managers\AmazonReportManager;
use SellingPartnerApi\ApiException;
use Throwable;

class AmazonReportController extends Controller
{
    /**
     * @throws Exception
     */
    public function getInfo(Request $request, AmazonIntegrationInstance $integrationInstance)
    {
        $request->validate([
            'reportType' => 'required|string|in:'.implode(',', AmazonReportTypeEnum::values()),
        ]);
        $reportType = AmazonReportTypeEnum::from($request->get('reportType'));
        return $this->response->addData((new AmazonLedgerManager($integrationInstance))->getInfo($reportType));
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     */
    public function sync(AmazonIntegrationInstance $integrationInstance, AmazonReport $report): Response
    {
        (new AmazonReportManager($integrationInstance))->sync($report);
        return $this->response->setMessage('Report synced');
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createReport(AmazonIntegrationInstance $integrationInstance, AmazonReportRequest $reportRequest): Response
    {
        $report = (new CreateAmazonReport($reportRequest))->handle();
        return $this->response->addData($report);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createNextRequest(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'reportType' => 'required|string|in:'.implode(',', AmazonReportTypeEnum::values()),
        ]);
        $reportType = AmazonReportTypeEnum::from($request->get('reportType'));
        $reportRequestData = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $integrationInstance,
            'report_type' => $reportType,
            'createImmediately' => false,
        ])))->handle();
        $reportRequest = $reportRequestData->createdRequests[0];
        return $this->response->addData($reportRequest);
    }

    /**
     * @throws Throwable
     */
    function getFbaInventoryValuation(AmazonIntegrationInstance $integrationInstance, Request $request): Response
    {
        $reportType = AmazonReportTypeEnum::FBA_INVENTORY_VALUATION;
        $reportRequestData = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $integrationInstance,
            'report_type' => $reportType,
            'createImmediately' => false,
        ])))->handle();
        $reportRequest = $reportRequestData->createdRequests[0];
        return $this->response->addData($reportRequest);
    }
}
