<?php

namespace Modules\Amazon\Http\Controllers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductController;
use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Exceptions\ExportingEmptyDataException;
use App\Exceptions\SalesChannelProductMappingException;
use App\Models\Product;
use App\Models\TaskStatus\TaskStatus;
use App\Response;
use Exception;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\InvalidSalesChannelProductMappingException;
use Modules\Amazon\Http\Resources\AmazonProductResource;
use Modules\Amazon\Jobs\ImportProductMapping;
use Modules\Amazon\Managers\AmazonProductManager;
use Modules\Amazon\Managers\AmazonReportManager;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Throwable;

class AmazonProductController extends AbstractSalesChannelProductController
{
    public function __construct(
        private readonly AmazonProductRepository $amazonProductRepository,
    ) {
        parent::__construct();
    }

    protected function getModel(): string
    {
        return AmazonProduct::class;
    }

    protected function getResource(): string
    {
        return AmazonProductResource::class;
    }

    // Typehint $integrationInstance with the integration implementation
    public function indexForIntegration(Request $request, AmazonIntegrationInstance $integrationInstance)
    {
        return parent::abstractIndexForIntegration($request, $integrationInstance);
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonProduct $product): Response
    {
        return $this->abstractShow($product);
    }

    /**
     * @throws Throwable
     */
    public function createSkuProducts(CreateSkuProductsFromSalesChannelData $data, AmazonIntegrationInstance $integrationInstance): Response {
        return $this->abstractCreateSkuProducts($data, $integrationInstance);
    }

    /**
     * @throws Throwable
     */
    public function refresh(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonReportManager($integrationInstance))->requestCreateSync(AmazonReportTypeEnum::PRODUCTS);

        return $this->response->setMessage('Added to queue, products will be refreshed shortly.');
    }

    /**
     * @throws Throwable
     */
    public function mapProducts(SalesChannelProductToSkuProductMappingCollectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        try {
            $response = $this->abstractMapProducts($data, $integrationInstance);
        } catch (SalesChannelProductMappingException $e) {
            return $this->response->setMessage($e->getMessage())->setStatus(400);
        }
        return $response;
    }

    /**
     * @throws ExportingEmptyDataException
     */
    protected function exportForIntegration(Request $request, AmazonIntegrationInstance $integrationInstance): BinaryFileResponse|Response|RedirectResponse
    {
        return $this->abstractExportForIntegration($request, $integrationInstance);
    }

    public function getMappingInfo(Request $request, AmazonIntegrationInstance $integrationInstance): ?Response
    {
        return $this->response->addData(
            $this->amazonProductRepository->getProductListingsMappingsInfo($request, $integrationInstance)
        );
    }

    public function salesChannelProductMappingsValidation(array $mappings): void
    {
        foreach ($mappings as $_mapping)
        {
            $_mapping = $_mapping[0];
            $existingProduct = Product::where('sku', $_mapping['mapped_sku'])->first();

            if ($existingProduct) {
                $amazonProduct = AmazonProduct::where('seller_sku', $_mapping['listing_sku'])->first();
                $allowedTypes = [Product::TYPE_STANDARD, Product::TYPE_KIT, Product::TYPE_BLEMISHED];

                if ($amazonProduct->is_fba && !in_array($existingProduct->type, $allowedTypes)) {
                    throw new HttpResponseException(
                        response()->json(["Unable to map $amazonProduct->seller_sku to a $existingProduct->type product"], 402)
                    );
                } elseif (!in_array($existingProduct->type, array_merge($allowedTypes, [Product::TYPE_BUNDLE]))) {
                    throw new HttpResponseException(
                        response()->json(['message' => "Unable to map $amazonProduct->seller_sku to a $existingProduct->type product"], 402)
                    );
                }
            }
        };
    }

    /**
     * @throws Exception
     */
    public function importMappings(SalesChannelProductImportMappingData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        return $this->abstractImportMappings($data, $integrationInstance);
    }
}
