<?php

namespace Modules\Amazon\Http\Controllers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstanceController;
use App\DataTable\DataTable;
use App\DTO\PaymentTypeData;
use App\Models\PaymentType;
use App\Repositories\PaymentTypeRepository;
use App\Repositories\SalesChannelRepository;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Modules\Amazon\Data\StoreAmazonIntegrationData;
use Modules\Amazon\Data\UpdateAmazonIntegrationData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Http\Resources\AmazonIntegrationInstanceResource;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonIntegrationInstanceManager;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class AmazonIntegrationInstanceController extends AbstractSalesChannelIntegrationInstanceController
{
    use DataTable;

    protected AmazonIntegrationInstanceRepository $integrationInstanceRepository;

    protected SalesChannelRepository $salesChannelRepository;

    private AmazonReportRepository $reportRepository;

    private PaymentTypeRepository $paymentTypeRepository;

    public function __construct()
    {
        $this->integrationInstanceRepository = app(AmazonIntegrationInstanceRepository::class);
        $this->salesChannelRepository        = app(SalesChannelRepository::class);
        $this->reportRepository = app(AmazonReportRepository::class);
        $this->paymentTypeRepository = app(PaymentTypeRepository::class);

        parent::__construct();
    }

    public function show(AmazonIntegrationInstance $integrationInstance): ?Response
    {
        $integrationInstance->load([
            'salesChannel',
            'integration',
        ]);

        return $this->response->addData($this->getResource()::make($integrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function store(StoreAmazonIntegrationData $data)
    {
        $response = parent::abstractStore($data);

        $integrationInstance = $response->getOriginalData();

        $this->paymentTypeRepository->save(PaymentTypeData::from([
        'name' => PaymentType::PAYMENT_TYPE_AMAZON,
        ]));

        return $response->setStatus(201)->addAdditional(
            'redirect_url',
            (new AmazonClient($integrationInstance))->getRedirectUrl()
        );
    }

    /**
     * @throws Throwable
     */
    public function update(UpdateAmazonIntegrationData $data, AmazonIntegrationInstance $integrationInstance): ?Response
    {
        $updateResponse = parent::abstractUpdate($data, $integrationInstance);

        if (isset($updateResponse->integrationSettingsChanged['is_fba_enabled'])) {
            (new AmazonIntegrationInstanceManager($integrationInstance))->processAmazonFbaStateChange();
        }

        return $updateResponse->response->addData(AmazonIntegrationInstanceResource::make($integrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function destroy(AmazonIntegrationInstance $integrationInstance): Response
    {
        return parent::abstractDestroy($integrationInstance);
    }

    protected function getModel(): string
    {
        return AmazonIntegrationInstance::class;
    }

    protected function getResource(): string
    {
        return AmazonIntegrationInstanceResource::class;
    }
}
