<?php

namespace Modules\Amazon\Http\Controllers;

use App\Data\IdSelectionData;
use App\DataTable\DataTableResource;
use App\Response;
use Illuminate\Http\Request;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Http\Resources\AmazonFbaReportRemovalOrderResource;
use Modules\Amazon\Managers\AmazonRemovalOrderManager;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class AmazonFbaRemovalOrderController extends AbstractAmazonDataTableController
{
    protected function getModel()
    {
        return AmazonFbaReportRemovalOrder::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonFbaReportRemovalOrderResource::class;
    }

    private AmazonLedgerRepository $ledgers;

    public function __construct()
    {
        $this->ledgers = app(AmazonLedgerRepository::class);
        parent::__construct();
    }

    public function getUnlinked(Request $request, AmazonIntegrationInstance $integrationInstance)
    {
        $amazonFbaRemovalOrders = $this->ledgers->getUnlinkedRemovalOrders($integrationInstance);

        return $this->response->addData($amazonFbaRemovalOrders);
    }

    /**
     * @throws Throwable
     */
    public function process(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        $processedShipments = (new AmazonRemovalOrderManager($integrationInstance))->process($data->ids);
        return $this->response->setMessage('Removal orders processed')->addData($this->getResource()::collection($processedShipments));
    }

    /**
     * @throws Throwable
     */
    public function processAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonRemovalOrderManager($integrationInstance))->processUnprocessed();
        return $this->response->setMessage('Removal orders processed');
    }

    public function unprocess(IdSelectionData $data, AmazonIntegrationInstance $integrationInstance): Response
    {
        (new AmazonRemovalOrderManager($integrationInstance))->unprocess($data->ids);
        return $this->response->setMessage('Removal orders unprocessed');
    }

    public function markItemAsUnknown(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportRemovalOrder $removalOrder): Response
    {
        (new AmazonRemovalOrderManager($integrationInstance))->markItemAsUnknown($removalOrder);
        return $this->response->setMessage('Item marked as unknown');
    }

    public function unmarkItemAsUnknown(AmazonIntegrationInstance $integrationInstance, AmazonFbaReportRemovalOrder $removalOrder): Response
    {
        (new AmazonRemovalOrderManager($integrationInstance))->unmarkItemAsUnknown($removalOrder);
        return $this->response->setMessage('Item unmarked as unknown');
    }
}
