<?php

namespace Modules\Amazon\Http\Controllers;

use App\DataTable\DataTableResource;
use App\Response;
use Exception;
use Illuminate\Http\Request;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Exceptions\CantUnreconcileInitialInventoryWithReconciledLedgersException;
use Modules\Amazon\Http\Controllers\Concerns\AmazonExportableControllerTrait;
use Modules\Amazon\Http\Resources\AmazonFbaInitialInventoryResource;
use Modules\Amazon\Managers\AmazonInitialInventoryManager;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class AmazonFbaInitialInventoryController extends AbstractAmazonDataTableController
{
    use AmazonExportableControllerTrait;

    private AmazonReportRepository $amazonReportRepository;

    public function __construct()
    {
        $this->amazonReportRepository = app(AmazonReportRepository::class);
        parent::__construct();
    }

    protected function getModel(): string
    {
        return AmazonFbaInitialInventory::class;
    }

    protected function getResource(): DataTableResource|string
    {
        return AmazonFbaInitialInventoryResource::class;
    }

    public function show(AmazonIntegrationInstance $integrationInstance, AmazonFbaInitialInventory $initialInventory): ?Response
    {
        $initialInventory->load([
            'amazonFnskuProduct',
            'fifoLayer'
        ]);

        return $this->response->addData($this->getResource()::make($initialInventory));
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function reconcileAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->reconcileAllInitialInventory();
        return $this->response->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function reconcile(AmazonIntegrationInstance $integrationInstance, AmazonFbaInitialInventory $initialInventory): Response
    {
        $initialInventory = (new AmazonInitialInventoryManager($integrationInstance))->reconcile($initialInventory);
        return $this->response->addData($this->getResource()::make($initialInventory->load('amazonFnskuProduct')));
    }

    /**
     * @throws Throwable
     */
    public function unreconcileAll(AmazonIntegrationInstance $integrationInstance): Response
    {
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->unreconcileAllInitialInventory();
        return $this->response->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }
    
    /**
     * @throws Exception|Throwable
     */
    public function unreconcile(AmazonIntegrationInstance $integrationInstance, AmazonFbaInitialInventory $initialInventory): Response
    {
        try {
            $initialInventory = (new AmazonInitialInventoryManager($integrationInstance))->unreconcile($initialInventory);
        } catch (CantUnreconcileInitialInventoryWithReconciledLedgersException $e) {
            return $this->response->setMessage($e->getMessage())->setStatusCode(400);
        }
        return $this->response->addData($this->getResource()::make($initialInventory->load('amazonFnskuProduct')));
    }

    /**
     * @throws Exception
     */
    public function reconcileInitialInventories(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'ids.*' => 'required|exists:amazon_fba_initial_inventory,id',
        ]);
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->reconcileInitialInventories($request->get('ids'));
        return $this->response->setMessage('Initial inventory reconciliation processed for ' . count($initialInventories) . ' records')->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }

    /**
     * @throws Exception
     */
    public function unreconcileInitialInventories(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'ids.*' => 'required|exists:amazon_fba_initial_inventory,id',
        ]);
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->unreconcileInitialInventories($request->get('ids'));
        return $this->response->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }

    /**
     * @throws Exception
     */
    public function reconcileInitialInventoriesForFnskuProducts(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'ids.*' => 'required|exists:amazon_fnsku_products,id',
        ]);
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->reconcileInitialInventoryForFnskuProducts($request->get('ids'));
        return $this->response->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }

    /**
     * @throws Exception
     */
    public function unreconcileInitialInventoriesForFnskuProducts(Request $request, AmazonIntegrationInstance $integrationInstance): Response
    {
        $request->validate([
            'ids.*' => 'required|exists:amazon_fnsku_products,id',
        ]);
        $initialInventories = (new AmazonInitialInventoryManager($integrationInstance))->unreconcileInitialInventoryForFnskuProducts($request->get('ids'));
        return $this->response->addData(AmazonFbaInitialInventoryResource::collection($initialInventories));
    }
}
