<?php

namespace Modules\Amazon\Entities;

use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo as ComposhipsBelongsTo;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property int $amazon_report_id
 * @property int $amazon_financial_event_group_id
 * @property-read string $settlement_id
 * @property-read string $transaction_type
 * @property-read string $order_id
 * @property-read string $merchant_order_id
 * @property-read string $adjustment_id
 * @property-read string $shipment_id
 * @property-read string $marketplace_name
 * @property-read string $amount_type
 * @property-read string $amount_description
 * @property-read string $amount
 * @property-read string $fulfillment_id
 * @property-read string $posted_date
 * @property-read string $posted_date_time
 * @property-read string $order_item_code
 * @property-read string $merchant_order_item_id
 * @property-read string $merchant_adjustment_item_id
 * @property-read string $sku
 * @property-read string $quantity_purchased
 * @property-read string $promotion_id
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonReport $amazonReport
 * @property-read AmazonFinancialEventGroup $amazonFinancialEventGroup
 * @property-read AmazonReportSettlementTypeMapping $amazonReportSettlementTypeMapping
 * @property-read Carbon $created_at
 * @property ?Carbon $updated_at
 */
class AmazonReportSettlementData extends AbstractAmazonReport implements Filterable, Sortable
{
    use BulkImport;
    use Compoships;
    use HasFactory;
    use HasFilters;
    use HasSort;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
        'posted_date_time' => 'datetime',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function amazonReport(): BelongsTo
    {
        return $this->belongsTo(AmazonReport::class);
    }

    public function amazonFinancialEventGroup(): BelongsTo
    {
        return $this->belongsTo(AmazonFinancialEventGroup::class);
    }

    public function amazonOrder(): HasOne
    {
        return $this->hasOne(AmazonOrder::class, 'AmazonOrderId', 'order_id');
    }

    public function amazonReportSettlementTypeMapping(): ComposhipsBelongsTo
    {
        return $this->belongsTo(
            AmazonReportSettlementTypeMapping::class,
            ['integration_instance_id', 'transaction_type', 'amount_type', 'amount_description'],
            ['integration_instance_id', 'transaction_type', 'amount_type', 'amount_description']
        );
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function availableColumns()
    {
        return config('data_table.amazon.settlement_data.columns');
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function delete(): ?bool
    {
        $this->amazonReportSettlementTypeMapping()->delete();

        return parent::delete();
    }
}
