<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrder;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderRepositoryInterface;
use App\Casts\EncryptedArray;
use App\Data\AddressData;
use App\Data\CustomerData;
use App\Data\FinancialLineData;
use App\Data\OrderLinkData;
use App\DTO\FieldValueMappingDto;
use App\DTO\PaymentDto;
use App\DTO\SalesOrderDto;
use App\DTO\SalesOrderStatusesDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Http\Resources\Amazon\OrdersResource;
use App\Integrations\SalesChannelOrder;
use App\Models\Concerns\BulkImport;
use App\Models\PaymentType;
use App\Models\ProductListing;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Repositories\FinancialLineRepository;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetOrdersAdt;
use Modules\Amazon\Database\Factories\AmazonOrderFactory;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * Class AmazonOrder.
 *
 * @property int $id
 * @property array $json_object
 * @property int $integration_instance_id
 * @property-read string $AmazonOrderId
 * @property-read string $SellerOrderId
 * @property-read string $PurchaseDate
 * @property-read Carbon $PurchaseDateUtc
 * @property-read Carbon $LastUpdateDate
 * @property-read OrderStatusEnum $OrderStatus
 * @property-read string $FulfillmentChannel
 * @property-read string $SalesChannel
 * @property-read string $ShipServiceLevel
 * @property-read string $OrderCurrency
 * @property-read float $OrderTotal
 * @property-read int $NumberOfItemsShipped
 * @property-read int $NumberOfItemsUnshipped
 * @property-read string $PaymentMethod
 * @property-read string $MarketplaceId
 * @property-read string $ShipmentServiceLevelCategory
 * @property-read string $BuyerName
 * @property-read string $BuyerEmail
 * @property-read string $Name // new
 * @property-read string $AddressLine1 // new
 * @property-read string $AddressLine2 // new
 * @property-read string $City // new
 * @property-read string $StateOrRegion // new
 * @property-read string $PostalCode // new
 * @property-read string $CountryCode // new
 * @property-read string $Phone // new
 * @property-read string $OrderType
 * @property-read string $EarliestShipDate
 * @property-read string $LatestShipDate
 * @property-read string $EarliestDeliveryDate
 * @property-read string $LatestDeliveryDate
 * @property-read string $IsPrime
 * @property-read string $ReplacedOrderId
 * @property-read string $IsReplacementOrder
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read Collection|AmazonOrderItem[] $amazonOrderItems
 *
 * @mixin IdeHelperAmazonOrder
 */
class AmazonOrder extends AbstractSalesChannelOrder implements SalesChannelOrder // SalesChannelOrder to deprecate
{
    use BulkImport;
    use HasFactory;

    protected $table = 'amazon_orders';

    protected $casts = [
        'archived_at' => 'datetime',
        'PurchaseDateUtc' => 'datetime',
        'LatestShipDateUtc' => 'datetime',
        'LatestDeliveryDateUtc' => 'datetime',
        'EarliestShipDateUtc' => 'datetime',
        'OrderStatus' => OrderStatusEnum::class,
        'NumberOfItemsUnshipped' => 'integer',
        'NumberOfItemsShipped' => 'integer',
        'json_object' => 'array',
        'items' => 'array',
        'OrderTotal' => 'float',
        'ShippingAddress' => EncryptedArray::class,
        'BuyerInfo' => EncryptedArray::class,
        'errors' => 'array',
        'canceled_at' => 'datetime',
        'error_log' => 'string',
        'AmazonPurchaseDate' => 'datetime',
    ];

    // used for getting listing data
    const LINE_ITEMS_QUERY_ID = 'OrderItemId';

    const LINE_ITEMS_QUERY = 'items';

    const PRODUCT_LOCAL_QUERY_IDENTIFER = 'seller_sku'; // AmazonProduct table

    const PRODUCT_LOCAL_FORIEN_IDENTIFER = 'SellerSKU'; // AmazonOrderItem table

    public static function getUniqueId(): string
    {
        return 'AmazonOrderId';
    }

    public static function getTableUniqueId(): string
    {
        return 'AmazonOrderId';
    }

    public static function getLastModified(): string
    {
        return 'LastUpdateDate';
    }

    public static function getOrderDate(): string
    {
        return 'PurchaseDateUtc';
    }

    public static function getCurrency(): ?string
    {
        return 'OrderCurrency';
    }

    public function getIsFulfillable(): bool
    {
        return $this->FulfillmentChannel == 'MFN';
    }

    public static function getDateRestrictionExceptionMapping(): ?FieldValueMappingDto
    {
        return FieldValueMappingDto::from([
            'field' => 'FulfillmentChannel',
            'value' => 'AFN',
        ]);
    }

    public function getEncryptedBuyerInfoAttribute()
    {
        if ($buyerInfo = $this->getAttributes()['BuyerInfo']) {
            return json_decode($buyerInfo, true);
        }

        return null;
    }

    public static function getIntegrationName(): string
    {
        return 'amazon';
    }

    public static function getItemClassName(): string
    {
        return AmazonOrderItem::class;
    }

    /*
    |--------------------------------------------------------------------------
    | Global Scope
    |--------------------------------------------------------------------------
    */

    protected static function booted(): void
    {
        static::addGlobalScope('exclude_non_amazon', function ($query) {
            $query->where('SalesChannel', 'not like', 'Non-Amazon%');
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function amazonIntegrationInstance(): BelongsTo
    {
        return $this->integrationInstance();
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(AmazonOrderItem::class, 'amazon_order_id', 'id');
    }

    public function amazonFinancialShipmentEvent(): HasMany
    {
        return $this->hasMany(AmazonFinancialShipmentEvent::class, 'AmazonOrderId', 'AmazonOrderId');
    }

    /**
     * @return Factory<static>
     */
    public static function newFactory(): Factory
    {
        return AmazonOrderFactory::new();
    }

    public static function getResource()
    {
        return OrdersResource::class;
    }

    public function getSkuOrderLines(): array
    {
        $salesChannel = $this->integrationInstance->salesChannel;
        $lines = [];
        /** @link https://siberventures.atlassian.net/browse/SKU-1818 */
        $shippingLine = [
            'description' => 'Shipping',
            'amount' => 0,
            'quantity' => 1,
            'discount_allocation' => 0,
            'tax_allocation' => 0,
            'is_product' => false,
        ];

        foreach ($this->amazonOrderItems as $item) {
            $listing = ProductListing::with([])->where('listing_sku', $item['SellerSKU'])
                ->where('sales_channel_listing_id', $item['ASIN'])
                ->where('sales_channel_id', $salesChannel->id)
                ->where('is_fba', $this->FulfillmentChannel === 'AFN')
                ->first();

            $line = null;

            // Non-bundle products
            $line = [
                'SellerSKU' => $item['SellerSKU'], // only use when map listing
                'description' => $item['Title'],
                'product_id' => $listing ? $listing->product_id : null,
                'product_listing_id' => $listing ? $listing->id : null,
                'quantity' => $item['QuantityOrdered'],
                'sales_channel_line_id' => $item['OrderItemId'],
                'is_product' => true,
                'warehouse_id' => $this->getPriorityWarehouseId($listing, $item),
            ];

            if ($line) {
                if (isset($item['ItemPrice'])) {
                    $line['amount'] = $item['QuantityOrdered'] != 0 ? $item['ItemPrice']['Amount'] / $item['QuantityOrdered'] : 0;
                    //        $line['discount_value'] = $item['PromotionDiscount']['Amount'] ?? 0;
                    $line['tax_total'] = $item['ItemTax']['Amount'] ?? 0;
                }

                $lines[] = $line;
            }

            if (isset($item['ShippingPrice']) && ((float) $item['ShippingPrice']['Amount']) > 0) {
                $shippingLine['amount'] += $item['ShippingPrice']['Amount'];
                $shippingLine['discount_allocation'] += $item['ShippingDiscount']['Amount'];
                $shippingLine['tax_allocation'] += $item['ShippingTax']['Amount'];
            }

            if (isset($item['PromotionDiscount']) && ((float) $item['PromotionDiscount']['Amount']) > 0) {
                $lines[] = [
                    'description' => isset($item['PromotionIds']) && count($item['PromotionIds']) ? implode('|', $item['PromotionIds']) : $item['SellerSKU'].' - Discount',
                    'amount' => floatval($item['ShippingPrice']['Amount']) * -1,
                    'quantity' => 1,
                    'is_product' => false,
                ];
            }
        }

        if ($shippingLine['amount'] > 0) {
            $lines[] = $shippingLine;
        }

        return $lines;
    }

    public function findMatchingLines(Collection $orderLines, ProductListing $productListing)
    {
        return $orderLines->where('SellerSKU', $productListing->listing_sku);
    }

    public function getSkuShippingMethodId()
    {
        return $this->getSkuShippingMethod();
    }

    public function isFullyShipped(): bool
    {
        return $this->OrderStatus == OrderStatusEnum::STATUS_SHIPPED;
    }

    public function isPartiallyShipped(): bool
    {
        return $this->OrderStatus == OrderStatusEnum::STATUS_PARTIALLY_SHIPPED;
    }

    public function partiallyFulfill(SalesOrder $salesOrder)
    {
        $salesOrder->load('salesOrderLines');

        $amazonShippedLines = collect($this->items)->where('QuantityShipped', '>', 0);

        $fulfillmentLines = [];
        foreach ($amazonShippedLines as $line) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $line['OrderItemId']);

            if ($salesOrderLine->fulfilled_quantity < $line['QuantityShipped']) {
                $fulfillmentLines[$salesOrderLine->warehouse_id][] = [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $line['QuantityShipped'] - $salesOrderLine->fulfilled_quantity,
                ];
            }
        }

        if (! empty($fulfillmentLines)) {
            foreach ($fulfillmentLines as $warehouseId => $lines) {
                try {
                    $fulfillRequest = FulfillSalesOrderRequest::createFromCustom(
                        [
                            'fulfillment_type' => $salesOrder->is_fba ? SalesOrderFulfillment::TYPE_FBA : SalesOrderFulfillment::TYPE_MANUAL,
                            'warehouse_id' => $warehouseId,
                            'shipping_method_id' => $salesOrder->shipping_method_id,
                            'fulfilled_at' => $this->LastUpdateDate,
                            'fulfillment_lines' => $lines,
                        ],
                        'POST',
                        ['sales_order' => $salesOrder]
                    );

                    FulfillSalesOrderService::make($salesOrder)->fulfill($fulfillRequest);
                } catch (Throwable $exception) {
                    Log::debug("can't fulfill {$this->AmazonOrderId}: {$exception->getMessage()}", $exception instanceof ValidationException ? $exception->errors() : []);
                }
            }
        }
    }

    public static function getOrderDateAttributeName(): string
    {
        return 'PurchaseDateUtc';
    }

    /**
     * when sku sales order deleted
     */
    public function skuOrderDeleted(): void
    {
        $this->salesOrder->sales_channel_order_id = null;
        $this->salesOrder->sales_channel_order_type = null;
        $this->salesOrder->save();
        $this->archive();
    }

    private function getFulfilledAt(): ?Carbon
    {
        return $this->OrderStatus->value == OrderStatusEnum::STATUS_SHIPPED() ? $this->LastUpdateDateUtc : null;
    }

    public function getSalesOrderStatusesDto(): SalesOrderStatusesDto
    {
        return SalesOrderStatusesDto::from([
            'order_status' => $this->FulfillmentChannel == 'MFN' ? SalesOrder::STATUS_OPEN : SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => $this->FulfillmentChannel == 'MFN' ? SalesOrder::FULFILLMENT_STATUS_UNFULFILLED : SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            'payment_status' => in_array($this->OrderStatus, OrderStatusEnum::STATUSES_ACTIVE) ? SalesOrder::PAYMENT_STATUS_PAID : SalesOrder::PAYMENT_STATUS_UNPAID,
        ]);
    }

    public function getCustomerDto(): ?CustomerData
    {
        $buyerEmail = @$this->encrypted_buyer_info['BuyerEmail'];
        $buyerName = @$this->encrypted_buyer_info['BuyerName'];

        return ($buyerName || $buyerEmail) ? CustomerData::from([
            'name' => $buyerName ?? (
                $buyerEmail ?? null
            ),
            'email' => $buyerEmail,
            'zip' => @$this->ShippingAddress['PostalCode'],
            'address1' => @$this->ShippingAddress['AddressLine1'],
            'company' => null,
            'fax' => null,
            'sales_channel_origin_id' => $this->integrationInstance->salesChannel->id,
            'default_shipping_address' => (
                ($this->ShippingAddress && $this->ShippingAddress != 'null') ? AddressData::from([
                    'name' => @$this->ShippingAddress['Name'] ?? $buyerEmail,
                    'email' => $buyerEmail,
                    'zip' => @$this->ShippingAddress['PostalCode'],
                    'label' => 'Shipping Address',
                    'address1' => @$this->ShippingAddress['AddressLine1'] ?? 'Amazon Address Line 1',
                    'address2' => @$this->ShippingAddress['AddressLine2'],
                    'address3' => @$this->ShippingAddress['AddressLine3'],
                    'city' => @$this->ShippingAddress['City'],
                    'province' => @$this->ShippingAddress['StateOrRegion'],
                    'province_code' => @$this->ShippingAddress['StateOrRegion'],
                    'country' => @$this->ShippingAddress['CountryCode'],
                    'country_code' => @$this->ShippingAddress['CountryCode'],
                ]) : null),
            'default_billing_address' => ($this->ShippingAddress && $this->ShippingAddress != 'null') ? AddressData::from([
                'name' => @$this->ShippingAddress['Name'] ?? $buyerEmail,
                'email' => $buyerEmail,
                'zip' => @$this->ShippingAddress['PostalCode'],
                'label' => 'Shipping Address',
                'address1' => @$this->ShippingAddress['AddressLine1'] ?? 'Amazon Address Line 1',
                'address2' => @$this->ShippingAddress['AddressLine2'],
                'address3' => @$this->ShippingAddress['AddressLine3'],
                'city' => @$this->ShippingAddress['City'],
                'province' => @$this->ShippingAddress['StateOrRegion'],
                'province_code' => @$this->ShippingAddress['StateOrRegion'],
                'country' => @$this->ShippingAddress['CountryCode'],
                'country_code' => @$this->ShippingAddress['CountryCode'],
            ]) : null,
        ]) : null;
    }

    public function getDiscountTotal(): float
    {
        return $this->orderItems->reduce(function ($carry, AmazonOrderItem $orderItem) {
            return $carry + $orderItem->PromotionDiscount;
        }, 0);
    }

    public function getDiscountTaxTotal(): float
    {
        return $this->orderItems->reduce(function ($carry, AmazonOrderItem $orderItem) {
            return $carry + $orderItem->PromotionDiscountTax;
        }, 0);
    }

    public function getShippingTotal(): float
    {
        return $this->orderItems->reduce(function ($carry, AmazonOrderItem $orderItem) {
            return $carry + ($orderItem->ShippingPrice - $orderItem->ShippingDiscount);
        }, 0);
    }

    public function getShippingTaxTotal(): float
    {
        return $this->orderItems->reduce(function ($carry, AmazonOrderItem $orderItem) {
            return $carry + ($orderItem->ShippingTax - $orderItem->ShippingDiscountTax);
        }, 0);
    }

    /**
     * @throws Throwable
     */
    public function getSalesOrderDto(): SalesOrderDto
    {
        $orderStatuses = $this->getSalesOrderStatusesDto();

        $salesOrderDto = SalesOrderDto::from([
            'sales_order_number' => $this->AmazonOrderId,
            'currency_code' => $this->getCurrencyCode($this->OrderCurrency),
            'order_date' => $this->PurchaseDateUtc,
            'fulfillment_channel' => $this->FulfillmentChannel,
            'fulfilled_at' => $this->getFulfilledAt(),
            'ship_by_date' => $this->LatestShipDateUtc ?? null,
            'deliver_by_date' => $this->LatestDeliveryDateUtc ?? null,
            'shipping_method_id' => null, // AFN / (Carrier ID - MFN)
            'requested_shipping_method' => $this->ShipServiceLevel,
            'is_fba' => $this->FulfillmentChannel == 'AFN',
            'is_replacement_order' => $this->IsReplacementOrder == 'true',
            'order_status' => $orderStatuses->order_status,
            'fulfillment_status' => $orderStatuses->fulfillment_status,
            'payment_status' => $orderStatuses->payment_status,
            // TODO: Handle canceled at
            'sales_order_lines' => $this->getSalesOrderLines(),
            'financial_lines' => $this->getFinancialLines(),
            'sales_channel_order_type' => self::class,
            'sales_channel_order_id' => $this->id,
            'last_synced_from_sales_channel_at' => $this->updated_at,
        ]);

        if ($customer = $this->getCustomerDto()) {
            $salesOrderDto->customer = $customer;
        }

        if (in_array($this->OrderStatus, OrderStatusEnum::STATUSES_ACTIVE)) {
            $salesOrderDto->payment = PaymentDto::from([
                'payment_date' => $this->PurchaseDateUtc,
                'payment_type' => PaymentType::PAYMENT_TYPE_AMAZON,
                'sales_order_number' => $this->AmazonOrderId, //Used for making join
                'currency_code' => $this->getCurrencyCode($this->OrderCurrency),
                'amount' => $this->getSalesOrderLines()->sum('amount'),
            ]);
        }

        if ($this->IsReplacementOrder == 'true') {
            $salesOrderDto->order_link = OrderLinkData::from([
                'parent_type' => static::class,
                'parent_order_number' => $this->ReplacedOrderId,
                'child_type' => static::class,
                'link_type' => 'resend',
            ]);
        }

        return $salesOrderDto;
    }

    public function getFinancialLines(): DataCollection
    {
        $financialLines = parent::getFinancialLines();

        // TODO: iterate through product listings to see if they have $proforma_marketplace_cost_percentage set and if so use that, otherwise
        //  fallback on the marketplace setting.  Need a way to maintain the proforma marketplace percentage or fee though
        if ($marketplaceCostPercentage = @$this->integrationInstance->integration_settings['proforma_marketplace_cost_percentage']) {
            $financialLines[] = FinancialLineData::from([
                'sales_order_id' => $this->id,
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType(
                        'Marketplace Fee',
                        FinancialLineClassificationEnum::COST,
                        prorationStrategy: FinancialLineProrationStrategyEnum::REVENUE_BASED,
                        allocateToProducts: true
                    )->id,
                'description' => 'Amazon Marketplace Fee',
                'quantity' => 1,
                'amount' => $this->OrderTotal * ($marketplaceCostPercentage / 100),
                'tax_allocation' => 0,
                'allocate_to_products' => 1,
                'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED, // TODO: Maybe use specific line?
            ]);
        }

        return $financialLines;
    }

    public function getFinancialEvents(): DataCollection
    {
        $financialLines = parent::getFinancialLines();

        // TODO: iterate through product listings to see if they have $proforma_marketplace_cost_percentage set and if so use that, otherwise
        //  fallback on the marketplace setting.  Need a way to maintain the proforma marketplace percentage or fee though
        if ($marketplaceCostPercentage = @$this->integrationInstance->integration_settings['proforma_marketplace_cost_percentage']) {
            $financialLines[] = FinancialLineData::from([
                'sales_order_id' => $this->id,
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType(
                        'Marketplace Fee',
                        FinancialLineClassificationEnum::COST,
                        prorationStrategy: FinancialLineProrationStrategyEnum::REVENUE_BASED,
                        allocateToProducts: true
                    )->id,
                'description' => 'Amazon Marketplace Fee',
                'quantity' => 1,
                'amount' => $this->OrderTotal * ($marketplaceCostPercentage / 100),
                'tax_allocation' => 0,
                'allocate_to_products' => 1,
                'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED, // TODO: Maybe use specific line?
            ]);
        }

        return $financialLines;
    }

    public function availableColumns(): array
    {
        $this->availableColumns = [
            'id' => 'integer',
            'AmazonOrderId' => 'varchar',
            'sku_sales_order_id' => 'integer',
            'SellerOrderId' => 'varchar',
            'PurchaseDate' => 'varchar',
            'PurchaseDateUtc' => 'datetime',
            'AmazonPurchaseDate' => 'datetime',
            'LastUpdateDate' => 'varchar',
            'OrderStatus' => 'varchar',
            'error_log' => 'varchar',
            'FulfillmentChannel' => 'varchar',
            'SalesChannel' => 'varchar',
            'ShipServiceLevel' => 'varchar',
            'OrderCurrency' => 'varchar',
            'OrderTotal' => 'varchar',
            'NumberOfItemsShipped' => 'varchar',
            'PaymentMethod' => 'varchar',
            'MarketplaceId' => 'varchar',
            'BuyerName' => 'text',
            'BuyerEmail' => 'text',
            'PaymentMethodDetails' => 'varchar',
            'ShipmentServiceLevelCategory' => 'varchar',
            'OrderType' => 'varchar',
            'EarliestShipDate' => 'varchar',
            'EarliestShipDateUtc' => 'datetime',
            'LatestShipDate' => 'varchar',
            'LatestShipDateUtc' => 'datetime',
            'EarliestDeliveryDate' => 'varchar',
            'EarliestDeliveryDateUtc' => 'datetime',
            'LatestDeliveryDate' => 'varchar',
            'LatestDeliveryDateUtc' => 'datetime',
            'IsBusinessOrder' => 'varchar',
            'IsPrime' => 'varchar',
            'IsPremiumOrder' => 'varchar',
            'ReplacedOrderId' => 'varchar',
            'IsReplacementOrder' => 'varchar',
            'json_object' => 'longtext',
            'errors' => 'longtext',
            'archived_at' => 'datetime',
            'canceled_at' => 'datetime',
            'created_at' => 'timestamp',
            'updated_at' => 'timestamp',
            'ShippingAddress' => 'longtext',
            'BuyerInfo' => 'longtext',
        ];

        return $this->availableColumns;
    }

    public static function specialLabels(): array
    {
        return [
            'sku_sales_order_id' => 'SKU Order',
            'PurchaseDateUtc' => 'Order Date',
        ];
    }

    public function visibleColumns(): array
    {
        return [
            'AmazonOrderId',
            'sku_sales_order_id',
            'PurchaseDateUtc',
            'LastUpdateDate',
            'OrderStatus',
            'error_log',
            'FulfillmentChannel',
            'ShipServiceLevel',
            'BuyerName',
            'OrderCurrency',
            'OrderTotal',
        ];
    }

    /**
     * @throws Exception
     */
    public function manager(AmazonIntegrationInstance|IntegrationInstanceInterface $integrationInstance): SalesChannelOrderManagerInterface
    {
        return new AmazonOrderManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new AmazonGetOrdersAdt();
    }

    public static function repository(): SalesChannelOrderRepositoryInterface
    {
        return app(AmazonOrderRepository::class);
    }
}
