<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\IntegrationModelInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;
use Modules\Amazon\Enums\Entities\FbaInboundPlanStatusEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $inboundPlanId
 * @property string $name
 * @property string $sourceName
 * @property FbaInboundPlanStatusEnum status
 * @property string $lastUpdatedAt
 * @property string createdAt
 * @property array $shipments
 * @property array $sourceAddress
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read Collection|AmazonNewFbaInboundPlanItem[] $amazonNewFbaInboundPlanItems
 * @property-read Collection|AmazonNewFbaInboundShipment[] $amazonNewFbaInboundShipments
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 */
class AmazonNewFbaInboundPlan extends Model implements Filterable, IntegrationModelInterface, MapsExportableFields, Sortable, DataTableModelInterface, UniqueFieldsInterface
{
    use Compoships,
        DataTableModelTrait,
        HasFactory,
        HasFilters,
        HasSort;

    protected $guarded = [];

    protected $casts = [
        'status' => FbaInboundPlanStatusEnum::class,
        'shipments' => 'array',
        'sourceAddress' => 'array',
        'json_object' => 'array',
    ];

    public static function getUniqueField(): string
    {
        return 'inboundPlanId';
    }

    public static function getUniqueFields(): array
    {
        return ['integration_instance_id', 'inboundPlanId'];
    }


    public static function getExportableFields(): array
    {
        return [
            'id' => Exportable::makeExportableField('id', false),
            'inboundPlanId' => Exportable::makeExportableField('inboundPlanId', false),
            'name' => Exportable::makeExportableField('name', false),
            'status' => Exportable::makeExportableField('status', false),
            'lastUpdatedAt' => Exportable::makeExportableField('lastUpdatedAt', false),
            'createdAt' => Exportable::makeExportableField('createdAt', false),
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonNewFbaInboundPlanItems(): HasMany
    {
        return $this->hasMany(AmazonNewFbaInboundPlanItem::class);
    }

    public function amazonNewFbaInboundShipments(): HasMany
    {
        return $this->hasMany(AmazonNewFbaInboundShipment::class);
    }

    public function skuLink(): MorphTo
    {
        return $this->morphTo('sku_link');
    }

    public function ledgers(): \Awobaz\Compoships\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(
            AmazonFbaReportInventoryLedger::class,
            ['integration_instance_id', 'reference_id'],
            ['integration_instance_id', 'ShipmentId']
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Filterable
    |--------------------------------------------------------------------------
    */

    public function visibleColumns(): array
    {
        return [
            'id',
            'inboundPlanId',
            'name',
            'status',
            'lastUpdatedAt',
            'createdAt',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['inboundPlanId', 'name', 'status'];
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Sortable
    |--------------------------------------------------------------------------
    */

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function delete()
    {
        // TODO: Update
        $this->amazonFbaInboundShipmentItems()->delete();

        return parent::delete();
    }
}
