<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstance;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductManager;
use App\Helpers;
use App\Models\ApiLog;
use App\Models\DataImportMapping;
use App\Models\Integration;
use App\Models\IntegrationShippingMethod;
use App\Models\InventoryMovement;
use App\Models\PaymentType;
use App\Models\SalesChannel;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\ShippingCarrier;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Managers\AmazonProductManager;
use Str;

/**
 * @property-read DataImportMapping $dataImportMapping
 *
 * @mixin IdeHelperAmazonIntegrationInstance
 */
class AmazonIntegrationInstance extends AbstractSalesChannelIntegrationInstance
{
    protected $table = 'integration_instances';

    protected $primaryKey = 'id';

    protected static function booted()
    {
        static::addGlobalScope('amazon', function (Builder $query) {
            return $query->whereHas('integration', function (Builder $query) {
                $query->where('name', Integration::NAME_AMAZON_US);
            });
        });
    }

    /**
     * @throws Exception
     */
    public function getLedgerStartDate(): Carbon
    {
        if (!@$this->integration_settings['fba_inventory_tracking_start_date'])
        {
            throw new Exception('FBA Inventory Tracking Start Date is not set');
        }
        return Carbon::parse($this->integration_settings['fba_inventory_tracking_start_date']);
    }

    /**
     * @throws Exception
     */
    public function getTimezone(): string
    {
        $regionSettings = config('amazon.regions');
        return $regionSettings[$this->country]['timezone'];
    }

    public function fbaInventoryTrackingStartDate(): ?Carbon
    {
        return $this->integration_settings['fba_inventory_tracking_start_date'] ?
            Carbon::parse($this->integration_settings['fba_inventory_tracking_start_date']) : null;
    }

    /**
     * @throws Exception
     */
    public function getFbaFulfillmentChannel(): string
    {
        return 'AMAZON_' . $this->country;
    }

    public static function getOrderClass(): string
    {
        return AmazonOrder::class;
    }

    public static function getOrderLineClass(): string
    {
        return AmazonOrderItem::class;
    }

    public static function getProductClass(): string
    {
        return AmazonProduct::class;
    }

    public function getMarketplaceIds(): array
    {
        $participatingMarketplaces = collect($this->integration_settings['marketplaceParticipations'])
            ->filter(function ($marketplace) {
                return Str::startsWith($marketplace['marketplace']['domainName'], 'www') &&
                    $marketplace['participation']['isParticipating'];
            });

        return $participatingMarketplaces
        ->map(fn ($marketplace) => $marketplace['marketplace']['id'])
        ->values()
        ->toArray();
    }

    public function getMarketplaceIdsCsv(): string
    {
        return implode(',', $this->getMarketplaceIds());
    }

    /**
     * @throws Exception
     */
    public function getProductManager(): AbstractSalesChannelProductManager
    {
        return new AmazonProductManager($this);
    }


    /*
     |--------------------------------------------------------------------------
     | Relations
     |--------------------------------------------------------------------------
     */

    public function apiLogs(): HasMany
    {
        return $this->hasMany(ApiLog::class, 'integration_instance_id');
    }

    public function dataImportMapping(): HasOne
    {
        return $this->hasOne(DataImportMapping::class, 'integration_instance_id');
    }

    public function inboundShipments(): HasMany
    {
        return $this->hasMany(AmazonFbaInboundShipment::class, 'integration_instance_id');
    }

    public function fbaInventory(): HasMany
    {
        return $this->hasMany(AmazonFbaReportInventory::class, 'integration_instance_id');
    }

    public function fbaReportCustomerReturns(): HasMany
    {
        return $this->hasMany(AmazonFbaReportCustomerReturn::class, 'integration_instance_id');
    }

    public function fbaReportInventoryLedgers(): HasMany
    {
        return $this->hasMany(AmazonFbaReportInventoryLedger::class, 'integration_instance_id');
    }

    public function fnSkuProducts(): HasMany
    {
        return $this->hasMany(AmazonFnskuProduct::class, 'integration_instance_id');
    }

    public function fbaInitialInventories(): HasMany
    {
        return $this->hasMany(AmazonFbaInitialInventory::class, 'integration_instance_id');
    }

    public function fbaReportRemovalOrders(): HasMany
    {
        return $this->hasMany(AmazonFbaReportRemovalOrder::class, 'integration_instance_id');
    }

    public function fbaReportRemovalShipments(): HasMany
    {
        return $this->hasMany(AmazonFbaReportRemovalShipment::class, 'integration_instance_id');
    }

    public function fbaReportRestocks(): HasMany
    {
        return $this->hasMany(AmazonFbaReportRestock::class, 'integration_instance_id');
    }

    public function fbaReportShipments(): HasMany
    {
        return $this->hasMany(AmazonFbaReportShipment::class, 'integration_instance_id');
    }

    public function financialEventGroups(): HasMany
    {
        return $this->hasMany(AmazonFinancialEventGroup::class, 'integration_instance_id');
    }

    public function fulfillmentOrders(): HasMany
    {
        return $this->hasMany(AmazonFulfillmentOrder::class, 'integration_instance_id');
    }

    public function integrationShippingMethods(): HasMany
    {
        return $this->hasMany(IntegrationShippingMethod::class, 'integration_instance_id');
    }

    public function orders(): HasMany
    {
        return $this->hasMany(AmazonOrder::class, 'integration_instance_id');
    }

    public function products(): HasMany
    {
        return $this->hasMany(AmazonProduct::class, 'integration_instance_id');
    }

    public function reports(): HasMany
    {
        return $this->hasMany(AmazonReport::class, 'integration_instance_id');
    }

    public function reportRequests(): HasMany
    {
        return $this->hasMany(AmazonReportRequest::class, 'integration_instance_id');
    }

    public function salesChannel(): HasOne
    {
        return $this->hasOne(SalesChannel::class, 'integration_instance_id');
    }

    public function warehouse(): HasOne
    {
        return $this->hasOne(Warehouse::class, 'integration_instance_id')->where('type', Warehouse::TYPE_AMAZON_FBA);
    }

    public function settlementData(): HasMany
    {
        return $this->hasMany(AmazonReportSettlementData::class, 'integration_instance_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function delete(): void
    {
        set_time_limit(0);
        DB::transaction(function () {
            /*
             * Clear all related records
             *
             * Many other models get deleted but handled through corresponding models
             */
            $warehouse = $this->warehouse;
            customlog('amazon', 'Deleting Amazon Integration Instance '.$this->name);
            customlog('amazon', '...Deleting API Logs');
            $this->apiLogs()->delete();
            customlog('amazon', '...Deleting Import Mapping');
            $this->dataImportMapping()->delete();
            if ($warehouse) {
                customlog('amazon', '...Deleting Inbound Shipments and Warehouse Transfers');
                $this->inboundShipments->each(function (AmazonFbaInboundShipment $amazonFbaInboundShipment) {
                    $amazonFbaInboundShipment->amazonFbaInboundShipmentItems()->delete();
                });
                $this->inboundShipments()->delete();
                $warehouseTransfers = WarehouseTransfer::where('from_warehouse_id', $warehouse->id)->orWhere('to_warehouse_id', $warehouse->id);
                $warehouseTransferShipments = WarehouseTransferShipment::whereIn('warehouse_transfer_id', $warehouseTransfers->pluck('id'));
                $warehouseTransferLines = WarehouseTransferLine::whereIn('warehouse_transfer_id', $warehouseTransfers->pluck('id'));
                $warehouseTransferShipmentLines = WarehouseTransferShipmentLine::whereIn('warehouse_transfer_shipment_id', $warehouseTransferShipments->pluck('id'));
                $warehouseTransferShipmentReceipts = WarehouseTransferShipmentReceipt::whereIn('warehouse_transfer_shipment_id', $warehouseTransferShipments->pluck('id'));
                $warehouseTransferShipmentReceiptLines = WarehouseTransferShipmentReceiptLine::whereIn('warehouse_transfer_shipment_receipt_id', $warehouseTransferShipmentReceipts->pluck('id'));
                InventoryMovement::where('link_type', WarehouseTransferShipmentLine::class)->whereIn('link_id', $warehouseTransferShipmentLines->pluck('id'))->delete();
                InventoryMovement::where('link_type', WarehouseTransferShipmentReceiptLine::class)->whereIn('link_id', $warehouseTransferShipmentReceiptLines->pluck('id'))->delete();
                $warehouseTransferShipmentReceiptLines->delete();
                $warehouseTransferShipmentReceipts->delete();
                $warehouseTransferShipmentLines->delete();
                $warehouseTransferShipments->delete();
                $warehouseTransferLines->delete();
                $warehouseTransfers->delete();
                customlog('amazon', '...Deleting FBA Inventory');
                $this->fbaInventory()->delete();
                customlog('amazon', '...Deleting Report -  Customer Returns');
                $this->fbaReportCustomerReturns()->delete();
                customlog('amazon', '...Deleting Report - Inventory Ledger Summaries');
                $this->fbaInitialInventories()->delete();
                customlog('amazon', '...Deleting Movements, FIFO Layers, Report - Inventory Ledgers');
                $ledgers = $this->fbaReportInventoryLedgers()->orderBy('event_datetime', 'desc')->orderBy('quantity')->get();
                DB::delete('DELETE FROM inventory_movements WHERE warehouse_id = '.$this->warehouse->id);
                DB::delete('DELETE FROM fifo_layers WHERE warehouse_id = '.$this->warehouse->id);
                $ledgers->each(function (AmazonFbaReportInventoryLedger $ledger) {
                    customlog('amazon', '......Deleting Ledger '.$ledger->id.': '.$ledger->event_datetime.' - '.$ledger->asin.' - '.$ledger->msku.' - '.$ledger->quantity);
                    $ledger->delete();
                });
                customlog('amazon', '...Deleting Fnsku Products');
                $this->fnSkuProducts()->delete();
                customlog('amazon', '...Deleting Report - Removal Orders');
                $this->fbaReportRemovalOrders()->delete();
                customlog('amazon', '...Deleting Report - Removal Shipments');
                $this->fbaReportRemovalShipments()->delete();
                customlog('amazon', '...Deleting Report - Restocks');
                $this->fbaReportRestocks()->delete();
                customlog('amazon', '...Deleting Report - Shipments');
                $this->fbaReportShipments()->delete();
            }

            customlog('amazon', '...Deleting Settlement Type Mapping');
            AmazonReportSettlementTypeMapping::query()->where('integration_instance_id', $this->id)->delete();
            customlog('amazon', '...Deleting Settlement Data');
            $this->settlementData()->delete();
            customlog('amazon', '...Deleting Financial Event Groups');
            $this->financialEventGroups()->delete();
            customlog('amazon', '...Deleting Fulfillment Orders');
            $this->fulfillmentOrders()->delete();
            customlog('amazon', '...Deleting Integration Shipping Methods');
            $this->integrationShippingMethods()->delete();

            customlog('amazon', '...Deleting Order Items');
            AmazonOrderItem::whereHas('order', function ($query) {
                $query->where('integration_instance_id', $this->id);
            })
                ->delete();

            customlog('amazon', '...Deleting Orders');
            $orders = $this->orders();
            $orderLines = SalesOrderLine::whereIn('sales_order_id', $orders->pluck('id'));
            SalesOrderLineFinancial::whereIn('sales_order_line_id', $orderLines->pluck('id'))->delete();
            $orders->delete();
            // TODO: Delete Inventory Movements
            // TODO: Delete FIFO Layers - last inventory movement deletes fifo layer?
            customlog('amazon', '...Deleting Products');
            $this->products()->delete();

            customlog('amazon', '...Deleting Reports');
            $this->reports()->delete();
            customlog('amazon', '...Deleting Report Requests');
            $this->reportRequests()->delete();

            // TODO: FBA Initial Inventory?
            // TODO: Inventory Adjustment

            customlog('amazon', '...Deleting Payment Type');
            if (self::query()->count() === 1) {
                PaymentType::query()
                    ->where('name', PaymentType::PAYMENT_TYPE_AMAZON)
                    ->first()?->delete();
                // There should not be any payments since orders already got deleted and payments get deleted with them

             ShippingCarrier::query()
                 ->where('name', 'Amazon MCF')
                 ->first()?->delete();
            }

            customlog('amazon', '...Deleting Products (again)');
            $this->products->each->delete();
            customlog('amazon', '...Deleting Orders (again)');
            $this->orders->each->delete();

            customlog('amazon', '...Deleting Sales Channel');
            $this->salesChannel?->delete();

            customlog('amazon', '...Deleting Integration Instance');
            DB::delete('DELETE FROM integration_instances WHERE id = ?', [$this->id]);

            if ($warehouse) {
                customlog('amazon', '...Deleting FBA Warehouse');
                $errors = $warehouse->delete();
                if (is_array($errors)) {
                    throw new Exception('Error deleting FBA Warehouse: '.implode(', ', $errors));
                }
            }
        });
    }
}
