<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\UniqueFieldsInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Product;
use Awobaz\Compoships\Compoships;
use Awobaz\Compoships\Database\Eloquent\Relations\HasMany;
use Awobaz\Compoships\Database\Eloquent\Relations\HasOne;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\Pivot;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $fnsku
 * @property string $location
 * @property string $disposition
 * @property int $product_id
 * @property int $reconciled_quantity
 * @property array $potential_product_matches
 * @property-read Product $product
 * @property-read EloquentCollection|AmazonFbaReportInventoryLedger[] $amazonFbaReportInventoryLedgers
 * @property-read EloquentCollection|AmazonFbaReportInventoryLedgerSummary[] $amazonFbaReportInventoryLedgerSummaries
 * @property-read AmazonFbaInitialInventory $amazonFbaInitialInventory
 * @property Carbon $created_at
 * @property null|Carbon $updated_at
 */
class AmazonFnskuProduct extends Pivot implements UniqueFieldsInterface, DataTableModelInterface
{
    use Compoships;
    use DataTableModelTrait;
    use HasFactory;
    use HasFilters;
    use HasSort;

    protected $table = 'amazon_fnsku_products';

    protected $guarded = [];

    protected $casts = [
        'potential_product_matches' => 'array',
    ];

    public static function getUniqueFields(): array
    {
        return ['integration_instance_id', 'fnsku'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function amazonFbaReportInventoryLedgers(): HasMany
    {
        return $this->hasMany(AmazonFbaReportInventoryLedger::class, [
            'integration_instance_id',
            'fnsku',
            'country',
            'disposition'
        ],
        [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    public function amazonFbaInitialInventory(): HasOne
    {
        return $this->hasOne(AmazonFbaInitialInventory::class, [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ],
        [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    public function amazonFbaReportInventoryLedgerSummaries(): HasMany
    {
        return $this->hasMany(AmazonFbaReportInventoryLedgerSummary::class, [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ],
        [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public static function specialLabels(): array
    {
        return [
            'id' => 'ID',
            'fnsku' => 'FNSKU',
            'product' => 'SKU Product'
        ];
    }

    public function availableColumns(): array
    {
        return [
            'id' => 'bigint',
            'fnsku' => 'string',
            'product' => 'string',
            'reconciled_quantity' => 'integer',
            'initialInventory' => 'initialInventory',
            'location' => 'string',
            'disposition' => 'string',
            'potential_product_matches' => 'json',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
        ];
    }

    public function visibleColumns(): array
    {
        return [
            'id',
            'fnsku',
            'product',
            'reconciled_quantity',
            'initialInventory',
            'location',
            'disposition',
            'potential_product_matches',
            'created_at',
            'updated_at',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return [
            'fnsku',
            'location',
            'disposition',
        ];
    }
}
