<?php

namespace Modules\Amazon\Entities;

use App\Models\AccountingTransaction;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupAccountingStatusEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property AmazonFinancialEventGroupAccountingStatusEnum $accounting_status
 * @property-read string $FinancialEventGroupId
 * @property-read string $ProcessingStatus
 * @property-read string $FundTransferStatus
 * @property-read string $OriginalTotalCurrencyCode
 * @property-read float $OriginalTotalCurrencyAmount
 * @property-read string $ConvertedTotalCurrencyCode
 * @property-read float $ConvertedTotalCurrencyAmount
 * @property-read string $FundTransferDate
 * @property-read string $TraceId
 * @property-read string $AccountTail
 * @property-read string $BeginningBalanceCurrencyCode
 * @property-read float $BeginningBalanceCurrencyAmount
 * @property-read string $FinancialEventGroupStart
 * @property-read ?string $FinancialEventGroupEnd
 * @property-read Carbon $FinancialEventGroupStart_datetime
 * @property-read ?Carbon $FinancialEventGroupEnd_datetime
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonFinancialEvent[]|EloquentCollection $amazonFinancialEvents
 * @property-read AccountingTransaction $accountingTransaction
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFinancialEventGroup
 */
class AmazonFinancialEventGroup extends Model implements Filterable, Sortable
{
    use BulkImport;
    use HasFactory;
    use HasFilters;
    use HasSort;
    use PowerJoins;

    const BULK_THRESHOLD = 1000;

    protected $guarded = [];

    protected $casts
        = [
            'json_object' => 'array',
            'FinancialEventGroupStart_datetime' => 'datetime',
            'FinancialEventGroupEnd_datetime' => 'datetime',
            'accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::class,
        ];

    public function getParentAccountingTransaction(): ?AccountingTransaction {
        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonIntegrationInstance(): BelongsTo
    {
        return $this->integrationInstance();
    }

    public function accountingTransaction(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function amazonReportSettlementData(): HasMany
    {
        return $this->hasMany(AmazonReportSettlementData::class);
    }

    public function amazonFinancialEvents(): HasMany
    {
        return $this->hasMany(AmazonFinancialEvent::class);
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function availableColumns()
    {
        return config('data_table.amazon.financial_event_group.columns');
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
