<?php

namespace Modules\Amazon\Entities;

use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property-read string $request_date
 * @property-read string $order_id
 * @property-read string $shipment_date
 * @property-read Carbon $event_datetime
 * @property-read string $sku
 * @property-read string $fnsku
 * @property-read string $disposition
 * @property-read int $shipped_quantity
 * @property-read string $carrier
 * @property-read string $tracking_number
 * @property-read FbaRemovalOrderTypeEnum $removal_order_type
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonFbaReportRemovalOrder $amazonFbaReportRemovalOrder
 * @property-read AmazonReport $amazonReport
 * @property-read AmazonFbaReportInventoryLedger $ledger
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFbaReportRemovalShipment
 */
class AmazonFbaReportRemovalShipment extends AbstractAmazonFbaDetailReport implements Filterable, MapsExportableFields, Sortable
{
    use BulkImport;
    use Compoships;
    use HasFilters;
    use HasSort;

    protected $casts = [
        'json_object' => 'array',
        'event_datetime' => 'datetime',
        'removal_order_type' => FbaRemovalOrderTypeEnum::class,
    ];

    public function getDateField(): string
    {
        return 'shipment_date';
    }

    public static function getQuantityField(): string
    {
        return 'shipped_quantity';
    }

    public static function hasNegativeLedgerImpact(): bool
    {
        return true;
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public static function getExportableFields(): array
    {
        return [
            'id' => Exportable::makeExportableField('id', false),
            'event_datetime' => Exportable::makeExportableField('event_datetime', false),
            'request_date' => Exportable::makeExportableField('request_date', false),
            'order_id' => Exportable::makeExportableField('order_id', false),
            'sku' => Exportable::makeExportableField('sku', false),
            'fnsku' => Exportable::makeExportableField('fnsku', false),
            'disposition' => Exportable::makeExportableField('disposition', false),
            'shipped_quantity' => Exportable::makeExportableField('shipped_quantity', false),
            'carrier' => Exportable::makeExportableField('carrier', false),
            'tracking_number' => Exportable::makeExportableField('tracking_number', false),
            'removal_order_type' => Exportable::makeExportableField('removal_order_type', false),
            'created_at' => Exportable::makeExportableField('created_at', false),
            'updated_at' => Exportable::makeExportableField('updated_at', false),
        ];

    }

    public function amazonFbaReportRemovalOrder(): BelongsTo
    {
        return $this->belongsTo(AmazonFbaReportRemovalOrder::class, 'order_id', 'order_id')
            ->where('integration_instance_id', $this->integration_instance_id)
            ->where('fnsku', $this->fnsku);
    }

    public function reportType(): AmazonReportTypeEnum
    {
        return AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS;
    }

    public function amazonReport(): BelongsTo
    {
        return $this->belongsTo(AmazonReport::class);
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function availableColumns()
    {
        return config('data_table.amazon.fba_report_removal_shipment.columns');
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
