<?php

namespace Modules\Amazon\Entities;

use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\WarehouseTransferLine;
use Awobaz\Compoships\Compoships;
use Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property-read string $request_date
 * @property-read string $order_id
 * @property-read FbaRemovalOrderTypeEnum $order_type
 * @property-read string $order_status
 * @property-read string $last_updated_date
 * @property-read string $sku
 * @property-read string $fnsku
 * @property-read string $disposition
 * @property-read int $requested_quantity
 * @property-read int $cancelled_quantity
 * @property-read int $disposed_quantity
 * @property-read int $shipped_quantity
 * @property-read int $in_process_quantity
 * @property-read float $removal_fee
 * @property-read string $currency
 * @property-read string $errorLog
 * @property array $json_object
 * @property bool $is_unknown_item
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read WarehouseTransferLine $skuLink // TODO: add SupplierReturnLine model once created
 * @property-read AmazonProduct $amazonProduct
 * @property-read AmazonFnskuProduct $amazonFnskuProduct
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 */
class AmazonFbaReportRemovalOrder extends AbstractAmazonReport implements AmazonFbaDetailReportInterface, DataTableModelInterface
{
    use BulkImport;
    use Compoships;
    use DataTableModelTrait;
    use HasFactory;
    use HasFilters;
    use HasSort;

    const REPORT_TYPE = AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
        'order_type' => FbaRemovalOrderTypeEnum::class,
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Exception
     */
    public function amazonProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, [
            'integration_instance_id',
            'sku',
        ], [
            'integration_instance_id',
            'seller_sku',
        ])->where('amazon_products.fulfillment_channel', '!=', AmazonProductFulfillmentChannelEnum::DEFAULT);
    }

    public function amazonFnskuProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonFnskuProduct::class, [
            'integration_instance_id',
            'fnsku',
        ], [
            'integration_instance_id',
            'fnsku',
        ]);
    }

    public function skuLink(): MorphTo
    {
        return $this->morphTo('sku_link');
    }

    public function reportType(): AmazonReportTypeEnum
    {
        return AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS;
    }

    public static function specialLabels(): array
    {
        return [
            'order_id' => 'Order ID',
            'sku' => 'SKU',
            'fnsku' => 'FNSKU',
            'skuLink' => 'SKU Link',
            'errorLog' => 'Error Log',
            'is_unknown_item' => 'Unknown Item?',
        ];
    }

    public function availableColumns(): array
    {
        return [
            'id' => 'bigint',
            'order_id' => 'string',
            'order_type' => 'string',
            'order_status' => 'string',
            'skuLink' => 'skuLink',
            'errorLog' => 'string',
            'sku' => 'string',
            'fnsku' => 'string',
            'disposition' => 'string',
            'request_date' => 'datetime',
            'requested_quantity' => 'int',
            'cancelled_quantity' => 'int',
            'disposed_quantity' => 'int',
            'shipped_quantity' => 'int',
            'in_process_quantity' => 'int',
            'removal_fee' => 'float',
            'currency' => 'string',
            'is_unknown_item' => 'checkbox',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
        ];
    }

    public function visibleColumns(): array
    {
        return [
            'id',
            'order_id',
            'order_type',
            'order_status',
            'skuLink',
            'errorLog',
            'sku',
            'fnsku',
            'disposition',
            'request_date',
            'requested_quantity',
            'cancelled_quantity',
            'disposed_quantity',
            'shipped_quantity',
            'in_process_quantity',
            'removal_fee',
            'currency',
            'is_unknown_item',
            'created_at',
            'updated_at',
        ];
    }
}
