<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\UniqueFieldsInterface;
use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\InventoryAdjustment;
use App\Services\InventoryManagement\InventoryEvent;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOneThrough;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property int $amazon_product_id
 * @property int $sku_link_id
 * @property string $sku_link_type
 * @property string $errorLog
 * @property int $blocked_by_ledger_id
 * @property-read string $date
 * @property Carbon $event_datetime
 * @property-read string $fnsku
 * @property-read string $asin
 * @property-read string $msku
 * @property-read string $title
 * @property-read FbaInventoryLedgerReportEventTypeEnum $event_type
 * @property-read string $reference_id
 * @property int $quantity
 * @property-read string $fulfillment_center
 * @property-read string $disposition
 * @property-read string $reason
 * @property-read string $country
 * @property-read int $reconciled_quantity
 * @property-read int $unreconciled_quantity
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonFbaReportInventoryLedger $blockedByLedger
 * @property-read AmazonProduct $amazonProduct
 * @property-read AmazonFnskuProduct $amazonFnskuProduct
 * @property-read AmazonLedgerDetail[]|EloquentCollection $ledgerDetails
 * @property-read AbstractAmazonFbaDetailReport[]|EloquentCollection $details
 * @property-read InventoryEvent $skuLink
 * @property-read AmazonFbaReportInventoryLedgerSummary $ledgerSummary
 * @property ?Carbon $last_reconciliation_attempted_at
 * @property ?Carbon $reconciled_at
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFbaReportInventoryLedger
 */
class AmazonFbaReportInventoryLedger extends AbstractAmazonReport implements Filterable, MapsExportableFields, Sortable
{
    use BulkImport,
        Compoships,
        HasFactory,
        HasFilters,
        HasSort;

    protected $table = 'amazon_fba_report_inventory_ledger';

    protected $guarded = [];

    protected $casts = [
        'quantity' => 'int',
        'event_datetime' => 'datetime',
        'reconciled_at' => 'datetime',
        'json_object' => 'array',
        'event_type' => FbaInventoryLedgerReportEventTypeEnum::class,
    ];

    public function getDateField(): string
    {
        return 'date';
    }

    public static function getUniqueFields(): array
    {
        return ['integration_instance_id', 'checksum'];
    }

    protected static function boot()
    {
        parent::boot();

        static::addGlobalScope('excludeWhseTransfers', function (Builder $builder) {
            $builder->where('event_type', '!=', FbaInventoryLedgerReportEventTypeEnum::WarehouseTransfers->value);
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    // Relationship to pivot table
    public function ledgerDetails(): HasMany
    {
        return $this->hasMany(AmazonLedgerDetail::class, 'amazon_fba_report_inventory_ledger_id');
    }

//    /**
//     * Get all the detail records associated with the ledger through the pivot table.
//     *
//     * This method is a bit more complex as Laravel does not support MorphToMany directly through a custom pivot.
//     * You need to access the details via the pivot records. This can be done manually when accessing the relation
//     * or by defining an accessor to simplify access.
//     */
//    public function details(): HasMany
//    {
//        return $this->ledgerDetails()->with('detail');
//    }

    // custom accessor to directly access the details in a more straightforward way
    public function getDetailsAttribute()
    {
        return $this->ledgerDetails->pluck('detail');
    }

    /**
     * Get the associated customer return detail.  We assume for now only one customer return per ledger.
     */
    public function customerReturn(): HasOneThrough
    {
        return $this->hasOneThrough(
            AmazonFbaReportCustomerReturn::class,
            AmazonLedgerDetail::class,
            'amazon_fba_report_inventory_ledger_id', // Foreign key on AmazonLedgerDetail table
            'id', // Foreign key on AmazonFbaReportCustomerReturn table
            'id', // Local key on AmazonFbaReportInventoryLedger table
            'detail_id'  // Local key on AmazonLedgerDetail table
        )->where('detail_type', AmazonFbaReportCustomerReturn::class);
    }

    /**
     * Get the associated removal shipment detail.  We assume for now only one removal shipment per ledger.
     */
    public function removalShipment(): HasOneThrough
    {
        return $this->hasOneThrough(
            AmazonFbaReportRemovalShipment::class,
            AmazonLedgerDetail::class,
            'amazon_fba_report_inventory_ledger_id', // Foreign key on AmazonLedgerDetail table
            'id', // Foreign key on AmazonFbaReportRemovalShipment table
            'id', // Local key on AmazonFbaReportInventoryLedger table
            'detail_id'  // Local key on AmazonLedgerDetail table
        )->where('detail_type', AmazonFbaReportRemovalShipment::class);
    }

    /**
     * Get the associated shipments detail.  We assume many shipments per ledger.
     */
    public function shipments(): HasManyThrough
    {
        return $this->hasManyThrough(
            AmazonFbaReportShipment::class,
            AmazonLedgerDetail::class,
            'amazon_fba_report_inventory_ledger_id', // Foreign key on AmazonLedgerDetail table
            'id', // Foreign key on AmazonFbaReportCustomerReturn table
            'id', // Local key on AmazonFbaReportInventoryLedger table
            'detail_id'  // Local key on AmazonLedgerDetail table
        )->where('detail_type', AmazonFbaReportShipment::class);
    }

    public function skuLink(): MorphTo
    {
        return $this->morphTo();
    }

    public function amazonProduct(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, ['integration_instance_id', 'msku'], ['integration_instance_id', 'seller_sku']);
    }

    public function amazonFnskuProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonFnskuProduct::class, [
            'integration_instance_id',
            'fnsku',
            'country',
            'disposition'
        ],
        [
            'integration_instance_id',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    public function blockedByLedger(): BelongsTo
    {
        return $this->belongsTo(AmazonFbaReportInventoryLedger::class, 'blocked_by_ledger_id');
    }

    public function ledgerSummary(): BelongsTo
    {
        return $this->belongsTo(AmazonFbaReportInventoryLedgerSummary::class, [
            'integration_instance_id',
            'date',
            'fnsku',
            'country',
            'disposition'
        ],
        [
            'integration_instance_id',
            'date',
            'fnsku',
            'location',
            'disposition'
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeFilterAdjustmentMadeForMissingDetails(Builder $builder, array $relation, string $operator, $value, $conjunction): Builder
    {
        $ledgersIds = self::query()
            ->where('event_type', '!=', FbaInventoryLedgerReportEventTypeEnum::Adjustments)
            ->where('sku_link_type', InventoryAdjustment::class)
            ->pluck('id');

        if ($value) {
            return $builder->whereIn('id', $ledgersIds);
        } else {
            return $builder->whereNotIn('id', $ledgersIds);
        }
    }

    public function scopeFilterSkuLinkExists(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            return $builder->whereHas('skuLink');
        } else {
            return $builder->whereNull('sku_link_id');
        }
    }

    public function scopeFilterDetailReconciled(Builder $builder, $operator, $value, $conjunction): Builder
    {
        /*
         * If value is true, should return all ledgers that have detail reports summing up to the same quantity as the ledger.
         * If value is false, should return all ledgers that do not have detail reports or have detail reports summing up to less than the ledger quantity.
         */
        if ($value) {
            return $builder->where(function ($builder) {
                $builder->whereHas('ledgerDetails');
            });
        } else {
            return $builder->where(function ($builder) {
                $builder->whereDoesntHave('ledgerDetails');
            });
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Filterable
    |--------------------------------------------------------------------------
    */

    public function availableColumns(): array
    {
        return config('data_table.amazon.fba_report_inventory_ledger.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['event_type', 'msku', 'reference_id'];
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Sortable
    |--------------------------------------------------------------------------
    */

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function delete(): ?bool
    {
        if ($skuLink = $this->skuLink) {
            $skuLink->delete();
        }

        return parent::delete();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exportable::makeExportableField('id', false),
            'unreconciled_quantity' => Exportable::makeExportableField('unreconciled_quantity', false),
            'country' => Exportable::makeExportableField('country', false),
            'disposition' => Exportable::makeExportableField('disposition', false),
            'fulfillment_center' => Exportable::makeExportableField('fulfillment_center', false),
            'reference_id' => Exportable::makeExportableField('reference_id', false),
            'quantity' => Exportable::makeExportableField('quantity', false),
            'amazon_product' => Exportable::makeExportableField('amazon_product', false),
            'asin' => Exportable::makeExportableField('asin', false),
            'fnsku' => Exportable::makeExportableField('fnsku', false),
            'msku' => Exportable::makeExportableField('msku', false),
            'errorLog' => Exportable::makeExportableField('errorLog', false),
            'sku_reference' => Exportable::makeExportableField('sku_reference', false),
            'sku_link' => Exportable::makeExportableField('sku_link', false),
            'detail_report' => Exportable::makeExportableField('detail_report', false),
            'event_type' => Exportable::makeExportableField('event_type', false),
            'date' => Exportable::makeExportableField('date', false),
            'created_at' => Exportable::makeExportableField('created_at', false),
            'updated_at' => Exportable::makeExportableField('updated_at', false),
            'adjustment_made_for_missing_details' => Exportable::makeExportableField('adjustment_made_for_missing_details', false),
        ];
    }

    public function scopeReconcileSort(Builder $builder): Builder
    {
        return $builder
            ->orderByRaw('event_datetime')
            ->orderByRaw('cast(quantity as int) desc');
    }

    public function scopeUnreconcileSort(Builder $builder): Builder
    {
        return $builder
            ->orderByRaw('event_datetime desc')
            ->orderByRaw('cast(quantity as int)');
    }

    public function scopeBeforeEventDate(Builder $query, $date): Builder
    {
        return $query->where('event_datetime', '<=', Carbon::parse($date));
    }

    public function scopeFnskuSort(Builder $builder): Builder
    {
        return $builder->orderBy('fnsku')
            ->orderBy('disposition')
            ->orderBy('country');
    }
}
