<?php

namespace Modules\Amazon\Entities;

use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property-read string $return_date
 * @property-read Carbon $event_datetime
 * @property-read string $order_id
 * @property-read string $sku
 * @property-read string $asin
 * @property-read string $fnsku
 * @property-read string $product_name
 * @property-read int $quantity
 * @property-read string $fulfillment_center_id
 * @property-read string $detailed_disposition
 * @property-read string $reason
 * @property-read string $status
 * @property-read string $license_plate_number
 * @property-read string $customer_comments
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read AmazonReport $amazonReport
 * @property-read AmazonFbaReportInventoryLedger $ledger
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFbaReportCustomerReturn
 */
class AmazonFbaReportCustomerReturn extends AbstractAmazonFbaDetailReport implements Filterable, MapsExportableFields, Sortable
{
    use BulkImport;
    use Compoships;
    use HasFilters;
    use HasSort;

    public function getDateField(): string
    {
        return 'return_date';
    }

    public static function getQuantityField(): string
    {
        return 'quantity';
    }

    public static function hasNegativeLedgerImpact(): bool
    {
        return false;
    }


    public function reportType(): AmazonReportTypeEnum
    {
        return AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS;
    }

    public function amazonReport(): BelongsTo
    {
        return $this->belongsTo(AmazonReport::class);
    }

    public function amazonProduct(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, ['integration_instance_id', 'sku'], ['integration_instance_id', 'seller_sku']);
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Filterable
    |--------------------------------------------------------------------------
    */

    public function availableColumns(): array
    {
        return config('data_table.amazon.fba_report_customer_return.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['order_id', 'sku', 'asin', 'fnsku', 'fulfillment_center_id', 'reason', 'status', 'license_plate_number'];
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers for Sortable
    |--------------------------------------------------------------------------
    */

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exportable::makeExportableField('id', false),
            'event_datetime' => Exportable::makeExportableField('event_datetime', false),
            'order_id' => Exportable::makeExportableField('order_id', false),
            'sku' => Exportable::makeExportableField('sku', false),
            'asin' => Exportable::makeExportableField('asin', false),
            'fnsku' => Exportable::makeExportableField('fnsku', false),
            'product_name' => Exportable::makeExportableField('product_name', false),
            'quantity' => Exportable::makeExportableField('quantity', false),
            'fulfillment_center_id' => Exportable::makeExportableField('fulfillment_center_id', false),
            'detailed_disposition' => Exportable::makeExportableField('detailed_disposition', false),
            'reason' => Exportable::makeExportableField('reason', false),
            'status' => Exportable::makeExportableField('status', false),
            'license_plate_number' => Exportable::makeExportableField('license_plate_number', false),
            'customer_comments' => Exportable::makeExportableField('customer_comments', false),
            'created_at' => Exportable::makeExportableField('created_at', false),
            'updated_at' => Exportable::makeExportableField('updated_at', false),
        ];

    }
}
