<?php

namespace Modules\Amazon\Entities;

use App\Models\Concerns\BulkImport;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;

/**
 * @property int $id
 * @property int $amazon_fba_inbound_shipment_id
 * @property string $SellerSKU
 * @property string $FulfillmentNetworkSKU
 * @property int $QuantityShipped
 * @property int $QuantityReceived
 * @property int $QuantityInCase
 * @property string $ReleaseDate
 * @property array $json_object
 * @property-read AmazonFbaInboundShipment $amazonFbaInboundShipment
 * @property-read AmazonProduct $amazonProduct
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 */
class AmazonFbaInboundShipmentItem extends Model
{
    use BulkImport,
        Compoships,
        HasFactory;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
    ];

    const BULK_THRESHOLD = 999999;

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function amazonFbaInboundShipment(): BelongsTo
    {
        return $this->belongsTo(AmazonFbaInboundShipment::class);
    }

    /**
     * @throws Exception
     */
    public function amazonProduct(): BelongsTo
    {
        return $this->belongsTo(AmazonProduct::class, 'SellerSKU', 'seller_sku')
            ->selectRaw('amazon_products.*')
            ->join('amazon_fba_inbound_shipments', 'amazon_products.integration_instance_id', '=', 'amazon_fba_inbound_shipments.integration_instance_id')
            ->whereColumn(
            'amazon_fba_inbound_shipments.integration_instance_id', 'amazon_products.integration_instance_id'
            )
            ->where('amazon_products.fulfillment_channel', '!=', AmazonProductFulfillmentChannelEnum::DEFAULT);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function getFnskuProduct(): ?AmazonFnskuProduct
    {
        return AmazonFnskuProduct::with('product.productInventory')
            ->where('integration_instance_id', $this->amazonFbaInboundShipment->integration_instance_id)
            ->where('fnsku', $this->FulfillmentNetworkSKU)
            ->where('disposition', 'SELLABLE')
            ->where('location', 'US')
            ->first();
    }
}
