<?php

namespace Modules\Amazon\Entities;

use App\Abstractions\Integrations\IntegrationModelInterface;
use App\DataTable\Exports\DataTableExporter as Exportable;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\PurchaseOrder;
use App\Models\WarehouseTransfer;
use Awobaz\Compoships\Compoships;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Collection;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $ShipmentId
 * @property string $ShipmentName
 * @property string $ShipFromName
 * @property string $DestinationFulfillmentCenterId
 * @property string $ShipmentStatus
 * @property string $LabelPrepType
 * @property string $BoxContentsSource
 * @property string $errorLog
 * @property int $sku_link_id
 * @property string $sku_link_type
 * @property bool $is_before_initial_count // Needed due to https://github.com/amzn/selling-partner-api-docs/issues/2706
 * @property array $json_object
 * @property-read AmazonIntegrationInstance $integrationInstance
 * @property-read Collection|AmazonFbaInboundShipmentItem[] $amazonFbaInboundShipmentItems
 * @property-read AmazonInboundShipmentSourceInterface $skuLink
 * @property-read Collection|AmazonFbaReportInventoryLedger[] $ledgers
 * @property-read AmazonFbaInboundShipFromMapping $shipFromMapping
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 *
 * @mixin IdeHelperAmazonFbaInboundShipment
 */
class AmazonFbaInboundShipment extends Model implements Filterable, IntegrationModelInterface, MapsExportableFields, Sortable
{
    use Compoships,
        HasFactory,
        HasFilters,
        HasSort;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
    ];

    public static function getUniqueField(): string
    {
        return 'ShipmentId';
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public static function getExportableFields(): array
    {
        return [
            'id' => Exportable::makeExportableField('id', false),
            'ShipmentId' => Exportable::makeExportableField('ShipmentId', false),
            'ShipmentName' => Exportable::makeExportableField('ShipmentName', false),
            'ShipFromName' => Exportable::makeExportableField('ShipFromName', false),
            'DestinationFulfillmentCenterId' => Exportable::makeExportableField('DestinationFulfillmentCenterId', false),
            'ShipmentStatus' => Exportable::makeExportableField('ShipmentStatus', false),
            'LabelPrepType' => Exportable::makeExportableField('LabelPrepType', false),
            'BoxContentsSource' => Exportable::makeExportableField('BoxContentsSource', false),
            'SKU Link' => Exportable::makeExportableField('SKU Link', false),
        ];
    }

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(AmazonIntegrationInstance::class);
    }

    public function amazonFbaInboundShipmentItems(): HasMany
    {
        return $this->hasMany(AmazonFbaInboundShipmentItem::class);
    }

    public function skuLink(): MorphTo
    {
        return $this->morphTo('sku_link');
    }

    public function ledgers(): \Awobaz\Compoships\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(
            AmazonFbaReportInventoryLedger::class,
            ['integration_instance_id', 'reference_id'],
            ['integration_instance_id', 'ShipmentId']
        );
    }

    public function shipFromMapping(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(AmazonFbaInboundShipFromMapping::class,
            ['integration_instance_id', 'ShipFromName'],
            ['integration_instance_id', 'name']
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function availableColumns()
    {
        return config('data_table.amazon.fba_inbound_shipment.columns');
    }

    public function generalFilterableColumns(): array
    {
        return ['ShipmentId', 'ShipmentName', 'ShipFromName',
            'DestinationFulfillmentCenterId',
        ];
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function delete(): ?bool
    {
        $this->amazonFbaInboundShipmentItems()->delete();

        return parent::delete();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeFilterSkuLinkExists(Builder $builder, $operator, $value, $conjunction): Builder
    {
        if ($value) {
            $builder->whereNotNull('sku_link_id');
        } else {
            $builder->whereNull('sku_link_id');
        }

        return $builder;
    }
}
