<?php

namespace Modules\Amazon\Console;

use App\Helpers;
use Illuminate\Console\Command;
use Modules\Amazon\Actions\RequestAmazonFeed;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;
use Throwable;

class RequestAmazonFeedCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'amazon:feeds:request
                            {integrationInstanceId? : The Integration Instance ID for Amazon}
                            {feedType? : The Type of Feed (i.e.: INVENTORY)}
                            {--o|options=* : Other report options key value pair with = separator}
                            {--a|available : Return list of available feeds}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create an Amazon Feed Request.';

    /**
     * Execute the console command.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        $arguments = $this->data();

        if ($this->option('available')) {
            AmazonIntegrationInstance::query()
                ->select(['integration_instances.id', 'integrations.name'])
                ->join('integrations', 'integrations.id', 'integration_instances.integration_id')
                ->each(function ($instance) {
                    $this->line("$instance->id: $instance->name");
                });
            collect(AmazonFeedTypeEnum::values())->each(function ($type) {
                $this->line($type);
            });

            return;
        }

        if (! isset($arguments['integrationInstanceId'])) {
            $this->error('Missing integrationInstanceId');
        }

        if (! isset($arguments['feedType'])) {
            $this->error('Missing feedType');
        }

        if (! (new RequestAmazonFeed(
            $arguments['integrationInstanceId'],
            AmazonFeedTypeEnum::tryFrom($arguments['feedType']),
            $this->option('options') ? Helpers::array2multidimensional($this->option('options'), '=') : null,
        ))->handle(false)) {
            $this->warn('Existing feed in queue for '.$arguments['feedType'].', 
            so no action needed (Instance: '.$arguments['integrationInstanceId'].')');
        }
    }

    private function data(): array
    {
        return [
            'integrationInstanceId' => $this->argument('integrationInstanceId') ?? $this->choice('Integration Instance ID', AmazonIntegrationInstance::query()->pluck('id')->toArray()),
            'feedType' => $this->argument('feedType') ?? $this->choice('Feed Type', AmazonFeedTypeEnum::values()),
        ];
    }
}
