<?php

namespace Modules\Amazon\Console;

use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Artisan;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Testing\WithFaker;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Throwable;

class AmazonSeedCommand extends Command
{
    use WithFaker;

    public function __construct(private readonly FulfillmentManager $fulfillmentManager, private readonly AmazonOrderRepository $amazonOrderRepository)
    {
        $this->faker = $this->makeFaker();
        parent::__construct();
    }

    /**
     * The console command name.
     *
     * @var string
     */
    protected $signature = 'amazon:seed';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Seed Amazon Dummy Data.';

    /**
     * Execute the console command.
     *
     * @throws Throwable
     */
    public function handle(): void
    {
        Artisan::call('db:seed', ['--class' => 'Modules\Amazon\Database\Seeders\AmazonSeeder']);

        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::query()->first();

        (new AmazonOrderManager($amazonIntegrationInstance))->createSkuOrders([], true);

        $counter = 0;

        $mfnOrders = SalesOrder::query()
            ->whereHas('salesChannelOrder', function (Builder $query) {
                $query->whereHas('integrationInstance', function (Builder $query) {
                    $query->whereHas('integration', function (Builder $query) {
                        $query->where('name', 'Amazon US');
                    });
                });
                $query->where('FulfillmentChannel', 'MFN');
            })
            ->limit(3)
            ->get()
            ->each(function (SalesOrder $salesOrder) use (&$counter) {
                $counter++;
                $fulfillmentLines = [];
                $salesOrder->productLines->each(function (SalesOrderLine $salesOrderLine) use (&$fulfillmentLines) {
                    $fulfillmentLines[] = [
                        'sales_order_line_id' => $salesOrderLine->id,
                        'quantity' => $salesOrderLine->quantity,
                    ];
                });
                $this->fulfillmentManager->fulfill($salesOrder, [
                    'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                    'warehouse_id' => 1,
                    'fulfilled_at' => $counter == 1 ? now()->subDays(15) : now()->subDays(45),
                    'tracking_number' => $this->faker->numerify('#############'),
                    'fulfillment_lines' => $fulfillmentLines,
                ]);
            });

        $this->amazonOrderRepository->resetPiiDataForFulfilledOrders();
    }
}
