<?php

namespace Modules\Amazon\Actions;

use Carbon\Carbon;
use Carbon\CarbonInterval;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Data\RequestAmazonReportResponseData;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class RequestAmazonReport
{
    private AmazonReportRepository $reports;
    private array $createdRequests = [];
    private array $createdReports = [];

    public function __construct(public RequestAmazonReportData $data) {
        $this->reports = app(AmazonReportRepository::class);
    }

    /**
     * @throws Throwable
     */
    public function handle(): RequestAmazonReportResponseData
    {
        customlog('amazon', "Requesting amazon report: {$this->data->report_type->value}");

        if (!$this->canRequestReport()) {
            return RequestAmazonReportResponseData::from([]);
        }
        $this->setStartDate();
        $this->setOptions();
        $this->sanitizeEndDate();
        $this->setMarketplaceIds();

        if (empty($this->data->data_end_date)) {
            return RequestAmazonReportResponseData::from([]);
        }

        // Is splittable report
        if (array_key_exists($this->data->report_type->value, AmazonReportTypeEnum::SPLITTABLE_REPORTS)) {
            $this->createSplitRequests();
        } else {
            $this->createRequests();
        }

        if ($this->data->createImmediately) {
            $this->createReports();
        }

        return RequestAmazonReportResponseData::from([
            'createdRequests' => $this->createdRequests,
            'createdReports' => $this->createdReports,
        ]);
    }

    /**
     * This is needed because Amazon has notoriously inaccurate data for the first 72 hours the data
     * exists.  See this blog as reference: https://www.sku.io/amazon-fba-inventory-reconciliation-and-accounting-issue/
     */
    private function sanitizeEndDate(): void
    {
        /*
         * If the start date is > max end date, we don't process the report
         * If end date > max end date or if end date is not specified, we change the end date to be the max end date
         * If end date is less than the start date, we don't process the report
         */
        $delayHours = AmazonReportTypeEnum::REPORT_DELAY_HOURS[$this->data->report_type->value] ?? 0;
        $maxEndDate = Carbon::now()->subHours($delayHours);

        if (empty($this->data->data_end_date) || $this->data->data_end_date->greaterThan($maxEndDate)) {
            $this->data->data_end_date = $maxEndDate->copy();
        }

        if (
            $this->data->data_start_date->greaterThan($maxEndDate) ||
            $this->data->data_start_date->greaterThan($this->data->data_end_date) ||
            $this->data->data_start_date->toString() == $this->data->data_end_date->toString()
        )
        {
            $this->data->data_end_date = null;
        }
    }

    /**
     * @throws Throwable
     */
    protected function requestReport(?array $marketplaceIds = null): AmazonReportRequest
    {
        /*
         * TODO: Validate start date where needed, for example for FBA_REPORT_INVENTORY_LEDGER make sure the start date
         *  is no earlier than the fba_inventory_tracking_start_date (make sure timezone is comparable)
         */
        return $this->reports->requestExists(
            $this->data->amazonIntegrationInstance->id,
            $this->data->report_type,
            $this->data->data_start_date,
            $this->data->data_end_date,
            $marketplaceIds
        ) ?
            $this->reports->getExistingRequestForType(
                $this->data->amazonIntegrationInstance->id,
                $this->data->report_type,
                $marketplaceIds
            ) :
            $this->reports->saveRequest(
                $this->data->amazonIntegrationInstance->id,
                $this->data->report_type,
                $this->data->data_start_date,
                $this->data->data_end_date,
                $this->data->options,
                $marketplaceIds
            );
    }

    private function setStartDate(): void
    {
        if (!$this->data->data_start_date) {
            $this->data->data_start_date = $this->reports->getStartDateForReportType($this->data->amazonIntegrationInstance,
                $this->data->report_type) ?? Carbon::parse($this->data->amazonIntegrationInstance->integration_settings['start_date']);
        }
    }

    private function setOptions(): void
    {
        if ($this->data->report_type === AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY) {
            $this->data->options = [
                'aggregatedByTimePeriod' => 'DAILY',
            ];
        }
    }

    private function setMarketplaceIds(): void
    {
        if (!$this->data->marketplace_ids) {
            $this->data->marketplace_ids = $this->data->amazonIntegrationInstance->getMarketplaceIds();
        }
    }

    /**
     * @throws Throwable
     */
    private function createSplitRequests(): void
    {
        $splitDays = AmazonReportTypeEnum::SPLITTABLE_REPORTS[$this->data->report_type->value];
        $endDate   = $this->data->data_end_date;

        $interval = CarbonInterval::days($splitDays);

        $dateRange = $endDate ?
            Carbon::parse($this->data->data_start_date)->toPeriod($endDate, $interval) :
            Carbon::parse($this->data->data_start_date)->toPeriod(Carbon::now(), $interval);

        foreach ($dateRange as $date) {
            $this->data->data_end_date = Carbon::parse($date->copy()->addDays($splitDays)->subSecond());
            $this->sanitizeEndDate();
            if (empty($this->data->data_end_date)) {
                continue;
            }
            $this->createdRequests[] = $this->requestReport();
        }
    }

    /**
     * @return void
     * @throws Throwable
     */
    private function createRequests(): void
    {
        if (in_array($this->data->report_type->value,
                AmazonReportTypeEnum::MARKETPLACE_REPORTS) && $this->data->marketplace_ids) {
            foreach ($this->data->marketplace_ids as $marketplace_id) {
                $this->createdRequests[] = $this->requestReport([$marketplace_id]);
            }
        } else {
            $this->createdRequests[] = $this->requestReport();
        }
    }

    /**
     * @return void
     * @throws Throwable
     */
    private function createReports(): void
    {
        foreach ($this->createdRequests as $createdRequest) {
            $this->createdReports[] = (new CreateAmazonReport($createdRequest))->handle();
        }
    }

    /**
     * Can only request report if there is not one already in process for that type
     * The exception is Settlement Report, which can be requested multiple times
     */
    private function canRequestReport(): bool
    {
        if (
            $this->reports->getLastReportInProcessForType($this->data->report_type, $this->data->amazonIntegrationInstance) &&
            $this->data->report_type !== AmazonReportTypeEnum::SETTLEMENT_REPORT
        ) {
            return false;
        }
        return true;
    }
}
