<?php

namespace Modules\Amazon\Actions\LedgerActions;

use App\DTO\InventoryAdjustmentDto;
use App\Exceptions\InsufficientStockException;
use App\Repositories\InventoryAdjustmentRepository;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Throwable;

class AmazonCreateInventoryAdjustmentFromLedger
{
    public function __construct(private readonly InventoryAdjustmentRepository $adjustments)
    {
    }

    /**
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function handle(
        AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger,
        string $notes,
        ?int $qtyOverride = null,
    ): AmazonFbaReportInventoryLedger
    {
        customlog('amazon', 'Creating inventory adjustment from ledger');

        $product = $amazonFbaReportInventoryLedger->amazonFnskuProduct->product;
        $quantity = $qtyOverride ?? $amazonFbaReportInventoryLedger->quantity;

        $data = InventoryAdjustmentDto::from([
            'adjustment_date' => Carbon::parse($amazonFbaReportInventoryLedger->event_datetime),
            'product_id' => $product->id,
            'warehouse_id' => $amazonFbaReportInventoryLedger->integrationInstance->warehouse->id,
            'quantity' => $quantity,
            'notes' => $notes,
        ]);

        $inventoryAdjustment = $this->adjustments->createInventoryAdjustment($data->toArray());

        $inventoryManager = InventoryManager::with(
            $inventoryAdjustment->warehouse_id,
            $inventoryAdjustment->product
        );
        if ($inventoryAdjustment->quantity > 0) {
            $inventoryManager->addToStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);
        } elseif ($inventoryAdjustment->quantity < 0) {
            $inventoryManager->takeFromStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);
        } else {
            throw new Exception('Inventory Adjustment quantity is 0 for amazon fba ledger ' . $amazonFbaReportInventoryLedger->id);
        }

        customlog('amazon', 'Clearing audit trail error');
        $amazonFbaReportInventoryLedger->skuLink()->associate($inventoryAdjustment);
        $amazonFbaReportInventoryLedger->last_reconciliation_attempted_at = Carbon::now();
        $amazonFbaReportInventoryLedger->reconciled_at = Carbon::now();
        $amazonFbaReportInventoryLedger->errorLog = null;
        $amazonFbaReportInventoryLedger->blocked_by_ledger_id = null;
        $amazonFbaReportInventoryLedger->save();
        $amazonFbaReportInventoryLedger->amazonFnskuProduct->reconciled_quantity += $amazonFbaReportInventoryLedger->quantity;
        $amazonFbaReportInventoryLedger->amazonFnskuProduct->save();

        return $amazonFbaReportInventoryLedger;
    }
}