<?php

namespace Modules\Amazon\Actions\LedgerActions;

use App;
use Carbon\Carbon;
use Closure;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\LedgerEventTypeWithNoDetailReportException;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;

class AmazonCheckIfDetailReportWasProcessed
{
    public function __construct(
        private readonly AmazonLedgerRepository $ledgers,
        private readonly AmazonReportRepository $reports
    ) {}

    /**
     * @throws LedgerEventTypeWithNoDetailReportException
     */
    public function handle(AmazonFbaReportInventoryLedger $ledger, Closure $next)
    {
        $hasDetailReport = in_array($ledger->event_type->value, FbaInventoryLedgerReportEventTypeEnum::HasDetailReports);
        customlog('amazon', 'Event Type: '.$ledger->event_type->value.' has detail? '.$hasDetailReport);
        if ($hasDetailReport) {
            //customlog('amazon', ' wasDetailReportProcessed? '.$this->wasDetailReportProcessed($ledger).'...');
        }

        if (! $ledger->details && $hasDetailReport) {
            customlog('amazon', 'Pending Detail Report Processing');
            $ledger->errorLog = 'Pending Detail Report Processing';
            $ledger->last_reconciliation_attempted_at = Carbon::now();
            $ledger->save();
            if (App::environment() != 'testing') {
                return $ledger;
            }
        }

        return $next($ledger);
    }

    /**
     * @throws LedgerEventTypeWithNoDetailReportException
     */
    private function wasDetailReportProcessed(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): bool
    {
        /*
         * We know the event type from the Amazon FBA Ledger Report.
         * From there we can get the detail report model.
         */
        $detailModel = $this->ledgers->getDetailModelFromLedgerEventType($amazonFbaReportInventoryLedger->event_type);
        $detailModel = app($detailModel);

        //customlog('amazon', $detailModel->reportType()->value . ' was it processed for ' . $amazonFbaReportInventoryLedger->event_datetime . '?');
        /*
         * We have to query amazon reports to make sure the reportType was requested and processed
         * For the date that is covered by the $amazonFbaReportInventoryLedger (must be in between $dataStartTime and $dataEndTime)
         * Note the event_datetime is not accurate because for example Vendor Returns have a report date that can vary greatly by event datetime (which is shipment_date in this case)...
         * So this is not a reliable check.
         */
        if (! $this->reports->wasReportTypeProcessedForDate($detailModel->reportType(), $amazonFbaReportInventoryLedger->event_datetime)) {
            return false;
        }

        /*
         * For Vendor Returns, we also need to make sure the Removal Orders report was processed
         */
        if ($amazonFbaReportInventoryLedger->event_type == FbaInventoryLedgerReportEventTypeEnum::VendorReturns) {
            if (! $this->reports->wasReportTypeProcessedForDate(AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS, $amazonFbaReportInventoryLedger->event_datetime)) {
                return false;
            }
        }

        return true;
    }
}