<?php

namespace Modules\Amazon\Actions;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use Exception;
use Modules\Amazon\Data\AmazonCreateOrderAcknowledgementFeedData;
use Modules\Amazon\Entities\AmazonFeedRequest;
use Modules\Amazon\Entities\AmazonFeedSubmission;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Managers\AmazonProductManager;
use Modules\Amazon\Repositories\AmazonFeedRepository;
use Modules\Amazon\Services\AmazonClient;

class CreateAmazonFeed extends AbstractSalesChannelManager
{
    protected AmazonFeedRepository $feedRepository;

    protected AmazonProductManager $productManager;

    protected AmazonOrderManager $amazonOrderManager;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonFeedRequest $amazonFeedRequest)
    {
        parent::__construct($this->amazonFeedRequest->integrationInstance, new AmazonClient($this->amazonFeedRequest->integrationInstance));

        $this->feedRepository = app(AmazonFeedRepository::class);
        $this->productManager = new AmazonProductManager($this->amazonFeedRequest->integrationInstance);
        $this->amazonOrderManager = new AmazonOrderManager($this->amazonFeedRequest->integrationInstance);
    }

    /**
     * @throws Exception
     */
    public function handle(): AmazonFeedSubmission
    {
        switch ($this->amazonFeedRequest->feedType) {
            case AmazonFeedTypeEnum::INVENTORY:
                $report = $this->feedRepository->create(
                    $this->amazonFeedRequest,
                    $this->client->createInventoryFeed(
                        $this->productManager->getProductListingsForInventoryFeed(),
                    )
                );

                break;

            case AmazonFeedTypeEnum::PRICING:
                $report = $this->feedRepository->create(
                    $this->amazonFeedRequest,
                    $this->client->createPricingFeed(
                        $this->productManager->getProductListingsForPricingFeed(),
                    )
                );

                break;

            case AmazonFeedTypeEnum::ORDER_ACKNOWLEDGEMENT:
                $report = $this->feedRepository->create(
                    $this->amazonFeedRequest,
                    $this->client->createAcknowledgeOrderFeed(
                        AmazonCreateOrderAcknowledgementFeedData::collection($this->amazonFeedRequest->metadata),
                    )
                );

                break;

            case AmazonFeedTypeEnum::ORDER_FULFILLMENT:
                $report = $this->amazonOrderManager->fulfillOrdersUsingAmazonFeed($this->amazonFeedRequest);
                break;
        }

        // After report is created we delete the request record
        $this->amazonFeedRequest->delete();

        return $report;
    }
}
