<?php

use App\Notifications\MonitoringMessage;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Notification;

function printLog($message): void
{
    if (config('app.debug')) {
        echo $message."\n";
    }
}

function arrayToTsv($array): string
{
    // Check if the array is empty
    if (empty($array)) {
        return '';
    }

    // Extract headers (keys of the first array element)
    $headers = array_keys($array[0]);
    $tsv = implode("\t", $headers)."\n";

    // Iterate over each row
    foreach ($array as $row) {
        $tsvRow = [];

        // Iterate over each cell
        foreach ($headers as $header) {
            $value = $row[$header] ?? ''; // Use null coalescing in case some headers are missing in some rows
            $tsvRow[] = str_replace(["\t", "\n", "\r"], ' ', $value); // Replace special characters to avoid breaking TSV format
        }

        // Join all cells with a tab and add to TSV string
        $tsv .= implode("\t", $tsvRow)."\n";
    }

    return $tsv;
}

function tsvToArray(
    $contents,
    $options = [
        'ignoreMissingColumns' => false,
        'trimColumns' => false,
    ]
) {
    $args = [];

    //key => default
    $fields = [
        'header_row' => true,
        'remove_header_row' => true,
        'trim_headers' => true, //trim whitespace around header row values
        'trim_values' => true, //trim whitespace around all non-header row values
        'lb' => "\n", //line break character
        'tab' => "\t", //tab character
    ];
    foreach ($fields as $key => $default) {
        if (array_key_exists($key, $args)) {
            $$key = $args[$key];
        } else {
            $$key = $default;
        }
    }

    $data = [];

    $lines = explode($lb, $contents);

    $row = 0;
    foreach ($lines as $line) {
        if (empty($line)) {
            continue;
        }
        $row++;
        if (($header_row) && ($row == 1)) {
            $data['headers'] = [];
        } else {
            $data[$row] = [];
        }
        $values = explode($tab, $line);

        foreach ($values as $c => $value) {
            if (($header_row) && ($row == 1)) { //if this is part of the header row
                if (in_array($value, $data['headers'])) {
                    throw new Exception('There are duplicate values in the header row: '.htmlspecialchars($value).'.', 1);
                } else {
                    if ($trim_headers) {
                        $value = trim($value);
                    }

                    $data['headers'][$c] = trim(strtolower(str_replace(['-', ' ', '/'], '_', strval($value))), '"');
                    /*
                     * Decided to keep old format for now
                     */
                    //$numeric_underscore_pattern = "/([_])([0-9])/";
                    /*
                     * Replaces example_field_1 with example_field1
                     */
                    //$data['headers'][$c] = preg_replace($numeric_underscore_pattern, "$2", $data['headers'][$c]);

                    $parentheses_pattern = "/[_]\(.*\)/";
                    $data['headers'][$c] = preg_replace($parentheses_pattern, '', $data['headers'][$c]);
                }
            } elseif ($header_row) { //if this isn't part of the header row, but there is a header row
                if ($options['ignoreMissingColumns']) {
                    $key = @$data['headers'][$c];
                } else {
                    $key = $data['headers'][$c];
                }

                if ($trim_values) {
                    $value = trim($value);
                    if ($options['trimColumns']) {
                        $value = trim($value, '"');
                    }
                }

                $data[$row][$key] = mb_convert_encoding($value, 'UTF-8');
            } else { //if there's not a header row at all
                $data[$row][$c] = mb_convert_encoding($value, 'UTF-8');
            }
        }
    }

    if ($remove_header_row) {
        unset($data['headers']);
    }

    return $data;
}

function encryptArray(?array $dataArray, ?array $keysToEncrypt = null)
{
    if ($dataArray) {
        foreach ($dataArray as $key => $value) {
            if (! is_null($keysToEncrypt)) {
                if (! is_null($keysToEncrypt) && in_array($key, $keysToEncrypt)) {
                    if (is_string($value)) {
                        $dataArray[$key] = encryptValue($value);
                    } elseif (is_array($value)) {
                        $dataArray[$key] = encryptArray($value);
                    }
                }
            } else {
                if (is_string($value)) {
                    $dataArray[$key] = encryptValue($value);
                } elseif (is_array($value)) {
                    $dataArray[$key] = encryptArray($value);
                }
            }
        }
    }

    return $dataArray;
}

function decryptArray(?array $dataArray, $test = false)
{
    if ($dataArray) {
        foreach ($dataArray as $key => $value) {
            if (is_array($value)) {
                $dataArray[$key] = decryptArray($value);
            } else {
                $dataArray[$key] = decryptValueIfEncrypted($value);
            }
        }
    }

    return $dataArray;
}

function encryptValue(mixed $value)
{
    if ($value) {
        return encryptAlgo($value);
        // return Crypt::encryptString($value);
    }
}

function decryptValue(mixed $value)
{
    if ($value) {
        return decryptAlgo($value);
        // return Crypt::decryptString($value);
    }
}

function decryptValueIfEncrypted($value)
{
    if (isEncrypted($value)) {
        return decryptValue($value);
    }

    return $value;
}

function isEncrypted(?string $encryptedString)
{
    if (empty($encryptedString)) {
        return false;
    }

    if (decryptValue($encryptedString) === false) {
        return false;
    } else {
        return true;
    }

    // try {
    //     decryptValue($encryptedString);

    //     return true;
    // } catch(\Illuminate\Contracts\Encryption\DecryptException $ex) {
    //     return false;
    // }
}

function getClassNameForBulkImport($class)
{
    return str_replace('\\', '\\\\', get_class($class));
}

/*
 * From: https://stackoverflow.com/questions/3876435/recursive-array-diff
 */
function arrayRecursiveDiff($aArray1, $aArray2): array
{
    $aReturn = [];

    foreach ($aArray1 as $mKey => $mValue) {
        if (array_key_exists($mKey, $aArray2)) {
            if (is_array($mValue)) {
                $aRecursiveDiff = arrayRecursiveDiff($mValue, $aArray2[$mKey]);
                if (count($aRecursiveDiff)) {
                    $aReturn[$mKey] = $aRecursiveDiff;
                }
            } else {
                if ($mValue != $aArray2[$mKey]) {
                    $aReturn[$mKey] = $mValue;
                }
            }
        } else {
            $aReturn[$mKey] = $mValue;
        }
    }

    return $aReturn;
}

function getRecordFromArray($array, $key, $value)
{
    return $array[array_search($value, array_column($array, $key))];
}

function array_unique_multidimensional($input): array
{
    return array_values(array_map('unserialize', array_unique(array_map('serialize', $input))));
}

function customlog($name, $message, $context = [], $days = 1): void
{
    $config = [
        'driver' => $days ? 'daily' : 'single',
        'level' => 'debug',
        'path' => storage_path("logs/$name.log"),
        'permission' => 0666,
    ];
    if ($days) {
        $config['days'] = $days;
    }
    Log::build($config)->debug($message, $context);
}

function slack($message): void
{
    $message = url('/').":\n\n ".$message;
    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($message));
}

function debug_summary(bool $includeDetails = false): array
{
    // Get the debug backtrace
    $backtrace = debug_backtrace();

    $summary = [];

    $appPath = base_path('app');
    // Filter the backtrace and keep only the files within your Laravel app directory
    // We apply array_filter to remove null values
    $filteredBacktrace = array_filter(array_map(function ($item) use ($appPath, &$summary) {
        if (isset($item['file']) && str_contains($item['file'], $appPath)) {
            $file = $item['file'];
            $line = $item['line'];
            $function = $item['function'] ?? '';
            $class = $item['class'] ?? '';
            $summary[] = basename($file).'->'.$function.'()'.':'.$line;

            return compact('file', 'line', 'function', 'class');
        }

        return [];
    }, $backtrace));

    $summary = ['Summary' => $summary];

    return $includeDetails ? array_merge($filteredBacktrace, $summary) : $summary;
}

// usage example: customlog('sales_order_line_observer', get_class($model) . ': ' . $model->id . "\n" .  debug_pretty_string());
function debug_pretty_string(): string
{
    return json_encode(debug_summary(), JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
}

function removeNullValuesFromArray(array $array): array
{
    return array_filter($array, function ($value) {
        return ! is_null($value);
    });
}

function toCsv(mixed $data): string
{
    return implode(',', Illuminate\Support\Arr::wrap($data));
}

function snakeString(string $string): string
{
    return str_replace(['-', ' ', '/'], '_', strtolower($string));
}

function catch_recursive($times): void
{
    static $run;
    $run++;
    if ($run > $times) {
        foreach (debug_backtrace() as $x) {
            echo $x['file'].':'.$x['line'].' -> '.($x['function'] ? $x['function'].'()' : '')."\n";
        }
        exit();
    }
}
function isGzipped($content)
{
    if (substr($content, 0, 2) == "\x1f\x8b") {
        return true; // The content is gzip compressed
    } else {
        return false; // The content is not gzip compressed
    }
}

// Initialize a potentially multidimensional array as 0 if it was not already set
function uninitializedArrayToZero(&$arr, ...$keys): void
{
    $temp = &$arr;

    foreach ($keys as $key) {
        if (! is_array($temp)) {
            $temp = [];
        }

        if (! isset($temp[$key])) {
            $temp[$key] = 0;
        }

        $temp = &$temp[$key];
    }
}

function safeDivide($numerator, $denominator, $default = 0): float
{
    return $denominator != 0 ? $numerator / $denominator : $default;
}

function deleteAllTemporaryTables()
{
    DB::table('information_schema.tables')
        ->select('table_name')
        ->where('table_name', 'like', 'temporary_table_%') // Your condition here
        ->where('table_name', '!=', 'temporary_tables')
        ->get()
        ->each(function ($table) {
            DB::statement("DROP TABLE IF EXISTS $table->table_name");
        });
}

function testEncryption(string $string)
{
    // Define your encryption key
    $key = 'your-encryption-key';

    // Define the supported cipher
    $cipher = 'AES-128-CBC'; // or 'AES-256-CBC'

    // Create an instance of the Encrypter class with the cipher and key
    $encrypter = new Encrypter($key, $cipher);

    // Encrypt your string
    $encryptedValue = $encrypter->encrypt($string);

    return $encryptedValue;
}

function consistentHash($value, $algorithm = 'sha256')
{
    return hash($algorithm, $value);
}

function encryptAlgo($string)
{
    $key = substr(config('app.key'), 0, 16);
    $iv = $key; // Fixed IV
    $encrypted = openssl_encrypt($string, 'AES-256-CBC', $key, 0, $iv);

    return base64_encode($encrypted);
}

function decryptAlgo($encryptedString)
{
    $key = substr(config('app.key'), 0, 16);
    $iv = $key; // Fixed IV
    $decrypted = openssl_decrypt(base64_decode($encryptedString), 'AES-256-CBC', $key, 0, $iv);

    return $decrypted;
}

function isFQN($string)
{
    // Regular expression to check for FQN format
    // This pattern requires at least one backslash as a namespace separator
    // It matches a series of alphanumeric characters and underscores, separated by backslashes
    $pattern = '/^[a-zA-Z0-9_]+(\\\\[a-zA-Z0-9_]+)+$/';

    // Use preg_match to check if the string matches the pattern
    return preg_match($pattern, $string);
}

function extractGraphQlId(string $gidString): ?string
{
    preg_match('/\d+$/', $gidString, $matches);

    return $matches[0] ?? null;
}

function isJson($string): bool
{
    json_decode($string);

    return json_last_error() == JSON_ERROR_NONE;
}

function getTailOfFQN($fqn): string
{
    // Get the last part of the namespace (the class name)
    return substr($fqn, strrpos($fqn, '\\') + 1);
}

function convertFQNToFriendlyName($fqn): string
{
    // Get the last part of the namespace (the class name)
    $className = getTailOfFQN($fqn);

    // Insert a space before each uppercase letter (except the first one)
    $friendlyName = preg_replace('/(?<!^)([A-Z])/', ' $1', $className);

    // Return the result with trimmed spaces
    return trim($friendlyName);
}

function xmlToArray(string $xmlString)
{
    $xml = simplexml_load_string($xmlString);

    // Convert SimpleXMLElement to an array
    $array = json_decode(json_encode($xml), true);

    return $array;
}

function friendlyName(string $string): string
{
    // Convert from snake case to human-readable and capitalize each word
    return ucwords(str_replace('_', ' ', $string));
}

function get_class_name_without_namespace($obj)
{
    return basename(str_replace('\\', '/', get_class($obj)));
}

function formatArrayToString($array, $indent = 0)
{
    $result = '';

    foreach ($array as $key => $value) {
        $result .= str_repeat('  ', $indent).ucfirst($key).":\n";

        if (is_array($value)) {
            $result .= formatArrayToString($value, $indent + 1);
        } else {
            $result .= str_repeat('  ', $indent + 1).$value."\n";
        }
    }

    return $result;
}

function getQueryDateFormat(string $interval): string
{
    // Define the date formats for each period
    $dateFormats = [
        'day' => '%Y-%m-%d',
        'week' => '%Y-%u',
        'month' => '%Y-%m',
        'quarter' => '%Y-%Q',
        'year' => '%Y',
    ];

    return $dateFormats[$interval] ?? $dateFormats['month'];
}
