<?php

namespace Tests\Utils;

use App\Models\InventoryMovement;
use App\Models\OrderLink;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

/*
 * This was a test used to understand the inner workings of factories and inheritance
 * When writing factories, we want to use clearly named methods in each factory as opposed to the
 * long form nested structure below.  This was simply an exercise to understand what is going on
 * in the background
 */

class FactoryTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_fully_received_purchase_order_factory_with_single_line(): void
    {
        PurchaseOrder::factory()
            ->has(
                PurchaseOrderLine::factory()
            )
            ->has(
                PurchaseOrderShipment::factory()
                    ->has(
                        PurchaseOrderShipmentLine::factory()
                            ->state(function ($attributes, PurchaseOrderShipment $purchaseOrderShipment) {
                                return [
                                    'purchase_order_line_id' => ($line = $purchaseOrderShipment->purchaseOrder->purchaseOrderLines()->first())->id,
                                    'quantity' => $line->quantity,
                                ];
                            })
                    )
                    ->has(
                        PurchaseOrderShipmentReceipt::factory()->has(
                            PurchaseOrderShipmentReceiptLine::factory()
                                ->state(function ($attributes, PurchaseOrderShipmentReceipt $purchaseOrderShipmentReceipt) {
                                    return [
                                        'purchase_order_shipment_line_id' => ($line = $purchaseOrderShipmentReceipt->purchaseOrderShipment->purchaseOrderShipmentLines()->first())->id,
                                        'quantity' => $line->quantity,
                                    ];
                                })
                        )
                    )
            )->create();

        $this->assertEquals(1, PurchaseOrder::all()->count());
        $this->assertDatabaseCount((new PurchaseOrder())->getTable(), 1);
    }

    public function test_fully_received_purchase_order_factory_with_multiple_lines(): void
    {
        $number_of_lines = 3;

        PurchaseOrder::factory()
            ->hasPurchaseOrderLines($number_of_lines)
            ->has(
                PurchaseOrderShipment::factory()
                    ->has(
                        PurchaseOrderShipmentLine::factory($number_of_lines)
                            ->state(new Sequence(
                                fn ($sequence) => [
                                    'purchase_order_line_id' => ($line = PurchaseOrderLine::offset($sequence->index)->first())->id,
                                    'quantity' => $line->quantity,
                                ]
                            )
                            )
                    )
                    ->has(
                        PurchaseOrderShipmentReceipt::factory()->has(
                            PurchaseOrderShipmentReceiptLine::factory($number_of_lines)
                                ->state(new Sequence(
                                    fn ($sequence) => [
                                        'purchase_order_shipment_line_id' => ($line = PurchaseOrderShipmentLine::offset($sequence->index)->first())->id,
                                        'quantity' => $line->quantity,
                                    ]
                                )
                                )
                        )
                    )
            )->create();

        $this->assertEquals(1, PurchaseOrder::all()->count());
        $this->assertEquals(3, PurchaseOrderLine::all()->count());
        $this->assertEquals(3, PurchaseOrderShipmentLine::all()->count());
        $this->assertEquals(3, PurchaseOrderShipmentReceiptLine::all()->count());
        $this->assertDatabaseCount((new PurchaseOrder())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderLine())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentLine())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceiptLine())->getTable(), 3);
    }

    public function test_fully_received_purchase_order_factory_with_multiple_lines_with_inventory_movements(): void
    {
        $number_of_lines = 3;

        PurchaseOrder::factory()
            ->hasPurchaseOrderLines($number_of_lines)
            ->has(
                PurchaseOrderShipment::factory()
                    ->has(
                        PurchaseOrderShipmentLine::factory($number_of_lines)
                            ->state(new Sequence(
                                fn ($sequence) => [
                                    'purchase_order_line_id' => ($line = PurchaseOrderLine::offset($sequence->index)->first())->id,
                                    'quantity' => $line->quantity,
                                ]
                            )
                            )
                    )
                    ->has(
                        PurchaseOrderShipmentReceipt::factory()->has(
                            PurchaseOrderShipmentReceiptLine::factory($number_of_lines)
                                ->state(new Sequence(
                                    fn ($sequence) => [
                                        'purchase_order_shipment_line_id' => ($line = PurchaseOrderShipmentLine::offset($sequence->index)->first())->id,
                                        'quantity' => $line->quantity,
                                    ]
                                )
                                )
                                ->has(
                                    InventoryMovement::factory()
                                        ->state(function ($attributes, PurchaseOrderShipmentReceiptLine $purchaseOrderShipmentReceiptLine) {
                                            return [
                                                'inventory_movement_date' => $purchaseOrderShipmentReceiptLine->purchaseOrderShipmentReceipt->received_at,
                                                'link_type' => PurchaseOrderShipmentReceiptLine::class,
                                                'link_id' => $purchaseOrderShipmentReceiptLine->id,
                                                'quantity' => $purchaseOrderShipmentReceiptLine->quantity,
                                                'type' => InventoryMovement::TYPE_PURCHASE_RECEIPT,
                                                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                                                'product_id' => $purchaseOrderShipmentReceiptLine->purchaseOrderShipmentLine->purchaseOrderLine->product->id,
                                            ];
                                        })
                                )
                        )
                    )
            )->create();

        $this->assertEquals(1, PurchaseOrder::all()->count());
        $this->assertEquals(3, PurchaseOrderLine::all()->count());
        $this->assertEquals(3, PurchaseOrderShipmentLine::all()->count());
        $this->assertEquals(3, PurchaseOrderShipmentReceiptLine::all()->count());
        $this->assertEquals(3, InventoryMovement::all()->count());
        $this->assertDatabaseCount((new PurchaseOrder())->getTable(), 1);
        $this->assertDatabaseCount((new PurchaseOrderLine())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentLine())->getTable(), 3);
        $this->assertDatabaseCount((new PurchaseOrderShipmentReceiptLine())->getTable(), 3);
        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 3);
    }

    public function test_fully_returned_sales_order_factory_with_multiple_lines_with_inventory_movements(): void
    {
        $number_of_lines = 3;

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines($number_of_lines)
            ->create();

        /** @var SalesCredit $salesCredit */
        $salesCredit = SalesCredit::factory()
            ->create([
                'sales_order_id' => $salesOrder->id,
            ]);

        SalesCreditLine::factory($number_of_lines)
            ->state(new Sequence(
                fn ($sequence) => [
                    'sales_order_line_id' => ($line = SalesOrderLine::offset($sequence->index)->first())->id,
                    'quantity' => $line->quantity,
                    'sales_credit_id' => SalesCredit::query()->first()->id,
                ]
            ))
            ->create();

        SalesCreditReturn::factory()
            ->has(
                SalesCreditReturnLine::factory($number_of_lines)
                    ->state(new Sequence(
                        fn ($sequence) => [
                            'sales_credit_line_id' => ($line = SalesCreditLine::offset($sequence->index)->first())->id,
                            'quantity' => $line->quantity,
                        ]
                    ))
                    ->has(
                        InventoryMovement::factory($number_of_lines)
                            ->state(function (array $attributes, SalesCreditReturnLine $salesCreditReturnLine) {
                                return [
                                    'link_id' => $salesCreditReturnLine->id,
                                    'link_type' => SalesCreditReturnLine::class,
                                    'type' => InventoryMovement::TYPE_RETURN,
                                    'quantity' => $salesCreditReturnLine->quantity,
                                ];
                            }))
            )->create(['sales_credit_id' => $salesCredit->id]);

        $this->assertEquals(1, SalesOrder::all()->count());
        $this->assertEquals(1, SalesCredit::all()->count());
        $this->assertEquals(1, OrderLink::all()->count());
        $this->assertEquals(3, SalesOrderLine::all()->count());
        $this->assertEquals(3, SalesCreditLine::all()->count());
        $this->assertEquals(3, SalesCreditReturnLine::all()->count());
        $this->assertEquals(9, InventoryMovement::all()->count());
    }
}
