<?php

namespace Tests\Unit\ShopifyOrders;

use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\Exceptions\ProductBundleException;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Address;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\Fulfillments\Dispatchers\StarshipitDispatcher;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Mockery\MockInterface;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class UpdateShopifyOrderTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private function makeShopifyOrder(): array
    {
        /** @var Address $address */
        $address = Address::factory()->create();

        /** @var Warehouse|null $warehouse */
        $warehouse = Warehouse::query()->first();
        $warehouse->update(['address_id' => $address->id]);

        $this->mock(StarshipitDispatcher::class, function (MockInterface $mock) {
            $mock->shouldReceive('dispatchFulfillmentToProvider');
        });

        /** @var Integration $starShipIt */
        $starShipIt = Integration::query()
            ->where('name', SalesOrderFulfillment::TYPE_STARSHIPIT)
            ->first();

        IntegrationInstance::factory()->create([
            'integration_id' => $starShipIt?->id,
            'integration_settings' => [
                'fulfillment' => [
                    'automatedWarehousesIds' => [$warehouse?->id],
                ],
            ],
        ]);

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'unfulfilled',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                        'is_product' => true,
                    ],
                    [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'product_exists' => true,
                        'quantity' => 3,
                        'name' => $this->faker->unique()->sentence(),
                        'variant_id' => 2,
                        'is_product' => true,
                    ],
                ],
                'customer' => [
                    'name' => $this->faker->name(),
                    'email' => $this->faker->email(),
                    'phone' => $this->faker->phoneNumber(),
                    'default_address' => [
                        'address1' => $this->faker->streetName(),
                        'zip' => $this->faker->postcode(),
                    ],
                ],
                'refunds' => [],
            ],
        ]);
        $shopifyOrder->save();
        $order = $shopifyOrder->createSKUOrder();

        return [$shopifyOrder, $warehouse];
    }

    public function test_it_wont_cancel_fulfilled_quantities(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        [$shopifyOrder, $warehouse] = $this->makeShopifyOrder();
        $jsonObject = $shopifyOrder->json_object;

        $jsonObject['fulfillments'][] = [
            'id' => 1,
            'status' => 'success',
            'tracking_company' => 'USPS',
            'tracking_numbers' => ['1Z2345'],
            'line_items' => [
                [
                    'id' => 2,
                    'quantity' => 2,
                    'sku' => 'SKU-2',
                    'variant_id' => 2,
                ],
            ],
            'created_at' => now()->subDay(),
            'updated_at' => now()->subDay(),
        ];

        $product = ProductListing::query()->where('sales_channel_listing_id', '2')->first()->product;

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 2,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $shopifyOrder->update(['json_object' => $jsonObject]);

        $shopifyOrder->updateSKUOrder();
        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
            'canceled_quantity' => 0,
            'externally_fulfilled_quantity' => 0,
        ]);

        // Attempt to cancel 3 units and only have 1 canceled since 2 is already fulfilled.
        $jsonObject['refunds'][] = [
            'id' => 1,
            'processed_at' => now()->subMinute(),
            'restock' => false,
            'refund_line_items' => [
                [
                    'id' => 1,
                    'line_item_id' => 2,
                    'quantity' => 3,
                    'restock_type' => 'no_restock',
                    'subtotal' => 15,
                    'line_item' => [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'variant_id' => 2,
                    ],
                ],
            ],
        ];

        $shopifyOrder->update(['json_object' => $jsonObject]);
        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 2,
            'canceled_quantity' => 1,
            'externally_fulfilled_quantity' => 0,
        ]);
    }

    public function test_it_can_fulfill_for_automated_fulfillment_warehouses(): void
    {
        Queue::fake();

        [$shopifyOrder, $warehouse] = $this->makeShopifyOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        $jsonObject = $shopifyOrder->json_object;
        $jsonObject['refunds'][] = [
            'id' => 1,
            'processed_at' => now()->subMinute(),
            'restock' => false,
            'refund_line_items' => [
                [
                    'id' => 1,
                    'line_item_id' => 2, // The second line is being removed
                    'quantity' => 3,
                    'restock_type' => 'no_restock',
                    'line_item' => [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'variant_id' => 2,
                    ],
                ],
            ],
        ];
        $jsonObject['fulfillments'][] = [
            'id' => 1,
            'status' => 'success',
            'tracking_company' => 'USPS',
            'tracking_numbers' => ['1Z2345'],
            'line_items' => [
                [
                    'id' => 1,
                    'quantity' => 2,
                    'sku' => 'SKU-1',
                    'variant_id' => 1,
                    'product_exists' => true,
                ],
            ],
            'created_at' => now()->subDay(),
            'updated_at' => now()->subDay(),
        ];

        $product = ProductListing::query()->where('sales_channel_listing_id', '1')->first()->product;

        Sanctum::actingAs(
            User::factory()->create(),
        );

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 5,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $shopifyOrder->update(['json_object' => $jsonObject]);

        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);

        /** @var SalesOrderLine $salesOrderLineDeleted */
        $salesOrderLineDeleted = SalesOrderLine::query()->where('sales_channel_line_id', 2)->first();
        // Queues are not working sometimes in tests, so we need to manually delete the backorder
        $salesOrderLineDeleted->backorderQueue?->delete();

        /*
         * 2 reservations
         * 1 adjustment
         * 1 fulfillment from the shopify sync
         */
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
            'canceled_quantity' => 0,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 0,
            'canceled_quantity' => 3,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);
    }

    public function test_it_can_update_shopify_order_with_refunds_and_externally_fulfilled_quantity(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        /** @var Address $address */
        $address = Address::factory()->create();

        /** @var Warehouse|null $warehouse */
        $warehouse = Warehouse::query()->first();
        $warehouse->update(['address_id' => $address->id]);

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'order_number' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'unfulfilled',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                        'is_product' => true,
                    ],
                    [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'product_exists' => true,
                        'quantity' => 3,
                        'name' => $this->faker->unique()->sentence(),
                        'variant_id' => 2,
                        'is_product' => true,
                    ],
                ],
                'refunds' => [],
            ],
        ]);
        $shopifyOrder->save();
        $shopifyOrder->createSKUOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        $jsonObject = $shopifyOrder->json_object;
        $jsonObject['refunds'][] = [
            'id' => 1,
            'processed_at' => now()->subMinute(),
            'restock' => false,
            'refund_line_items' => [
                [
                    'id' => 1,
                    'line_item_id' => 2, // The second line is being removed
                    'quantity' => 3,
                    'restock_type' => 'no_restock',
                    'line_item' => [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'variant_id' => 2,
                    ],
                ],
            ],
        ];
        $jsonObject['fulfillments'][] = [
            'id' => 1,
            'status' => 'success',
            'tracking_company' => 'USPS',
            'tracking_numbers' => ['1Z2345'],
            'line_items' => [
                [
                    'id' => 1,
                    'quantity' => 2,
                    'sku' => 'SKU-1',
                    'variant_id' => 1,
                    'product_exists' => true,
                ],
            ],
            'created_at' => now()->subDay(),
            'updated_at' => now()->subDay(),
        ];

        $product = ProductListing::query()->where('sales_channel_listing_id', '1')->first()->product;

        Sanctum::actingAs(
            User::factory()->create(),
        );

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => Carbon::now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5.00,
            'quantity' => 5,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();

        $shopifyOrder->update(['json_object' => $jsonObject]);

        $shopifyOrder->updateSKUOrder();

        /** @var SalesOrderLine $salesOrderLineDeleted */
        $salesOrderLineDeleted = SalesOrderLine::query()->where('sales_channel_line_id', 2)->first();
        // Queues are not working sometimes in tests, so we need to manually delete the backorder
        $salesOrderLineDeleted->backorderQueue?->delete();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
            'canceled_quantity' => 0,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 0,
            'canceled_quantity' => 3,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_it_can_update_shopify_order(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'unfulfilled',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                    ],
                    [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'product_exists' => true,
                        'quantity' => 3,
                        'name' => $this->faker->unique()->sentence(),
                        'variant_id' => 2,
                    ],
                ],
                'refunds' => [],
            ],
        ]);
        $shopifyOrder->save();
        $shopifyOrder->createSKUOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        // To reduce line item quantity, we create a refund.
        // That's how shopify reduces quantities.
        $jsonObject = $shopifyOrder->json_object;
        $jsonObject['refunds'][] = [
            'id' => 1,
            'processed_at' => now()->subMinute(),
            'restock' => false,
            'refund_line_items' => [
                [
                    'id' => 1,
                    'line_item_id' => 2, // The second line is being removed
                    'quantity' => 3,
                    'restock_type' => 'no_restock',
                    'line_item' => [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'variant_id' => 2,
                    ],
                ],
            ],
        ];
        $shopifyOrder->update(['json_object' => $jsonObject]);

        $shopifyOrder->updateSKUOrder();

        /** @var SalesOrderLine $salesOrderLineDeleted */
        $salesOrderLineDeleted = SalesOrderLine::query()->where('sales_channel_line_id', 2)->first();
        // Queues are not working sometimes in tests, so we need to manually delete the backorder
        $salesOrderLineDeleted->backorderQueue?->delete();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
            'canceled_quantity' => 0,
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 0,
            'canceled_quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        // Update shipping address
        $jsonObject = $shopifyOrder->json_object;

        $jsonObject['refunds'] = [];
        $jsonObject['customer'] = [
            'email' => 'rick@sku.io',
            'first_name' => 'Rick',
            'last_name' => 'Jacobs',
        ];
        $jsonObject['shipping_address'] = [
            'first_name' => 'Rick',
            'address1' => '500 Westover Drive',
            'phone' => '+17608808820',
            'city' => 'Sanford',
            'zip' => '27330',
            'province' => 'North Carolina',
            'country' => 'United States',
            'last_name' => 'Jacobs',
            'address2' => '#140-2671 4ax',
            'company' => '',
            'latitude' => 35.462585,
            'longitude' => -79.2063632,
            'name' => 'Rick Jacobs',
            'country_code' => 'US',
            'province_code' => 'NC',
        ];
        $shopifyOrder->json_object = $jsonObject;
        $shopifyOrder->save();
        $shopifyOrder->refresh();

        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new Address()), [
            'address2' => '#140-2671 4ax',
        ]);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_it_can_update_shopify_order_with_tax(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);
        $lineItem1 = $this->faker->sentence();
        $lineItem2 = $this->faker->sentence();
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'unfulfilled',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'price' => 8.7,
                        'quantity' => 5,
                        'name' => $lineItem1,
                        'variant_id' => 1,
                    ],
                    [
                        'id' => 2,
                        'sku' => 'SKU-2',
                        'product_exists' => true,
                        'price' => 7.5,
                        'quantity' => 3,
                        'name' => $lineItem2,
                        'variant_id' => 2,
                    ],
                ],
                'tax_lines' => [
                    [
                        'price' => 8.7,
                        'rate' => 0.0185,
                        'title' => $lineItem1,
                    ],
                    [
                        'price' => 7.5,
                        'rate' => 0.015,
                        'title' => $lineItem2,
                    ],
                ],
                'refunds' => [],
                'current_total_tax' => 6.83,
                'taxes_included' => false
            ],
        ]);
        $shopifyOrder->save();
        $salesOrder = $shopifyOrder->createSKUOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 4);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        // To reduce line item quantity, we create a refund.
        // That's how shopify reduces quantities.
        $jsonObject = $shopifyOrder->json_object;

        unset($jsonObject['current_total_tax']);

        $shopifyOrder->update(['json_object' => $jsonObject]);

        $shopifyOrder->updateSKUOrder();

        /** @var SalesOrderLine $salesOrderLineDeleted */
        $salesOrderLineDeleted = SalesOrderLine::query()->where('sales_channel_line_id', 2)->first();
        // Queues are not working sometimes in tests, so we need to manually delete the backorder
        $salesOrderLineDeleted->backorderQueue?->delete();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 1,
            'quantity' => 5,
            'canceled_quantity' => 0,
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'sales_channel_line_id' => 2,
            'quantity' => 3,
        ]);

        $this->assertDatabaseMissing('shopify_orders', [
            'sku_sales_order_id' => null,
        ]);

        $this->assertDatabaseMissing('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'tax_total' => null,
        ]);

        $this->assertGreaterThan(0, $salesOrder->tax_total);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_shopify_update_adds_second_line_item(): void
    {
        $newWarehouse = Warehouse::factory()->create();

        Sanctum::actingAs(User::first());

        $integrationInstance = SalesChannel::where('id', 1)->first()->integrationInstance;

        $shopifyProduct1 = ShopifyProduct::factory()->create([
            'integration_instance_id' => $integrationInstance->id,
        ]);
        $jsonObject = $shopifyProduct1->json_object;
        $jsonObject['sku'] = 'CC_LB-TL-FZ-5-6-8PACK-230824';
        $shopifyProduct1->update(['json_object' => $jsonObject]);

        $shopifyProduct2 = ShopifyProduct::factory()->create([
            'integration_instance_id' => $integrationInstance->id,
        ]);
        $jsonObject = $shopifyProduct1->json_object;
        $jsonObject['sku'] = 'REGTAILS4UPSELL';
        $shopifyProduct2->update(['json_object' => $jsonObject]);

        $bundle1 = Product::factory()->create([
            'sku' => 'CC_LB-TL-FZ-5-6-8PACK-230824',
            'type' => 'bundle',
        ]);

        $componentForBundle1 = Product::factory()->create([
            'sku' => 'LB-TL-FZ-5-6-4PACK',
        ]);

        $bundle1->setBundleComponents([
            [
                'id' => $componentForBundle1->id,
                'quantity' => 2,
            ],
        ]);

        $bundle2 = Product::factory()->create([
            'sku' => 'REGTAILS4UPSELL',
            'type' => 'bundle',
        ]);

        $componentForBundle2 = Product::factory()->create([
            'sku' => 'LB-TL-FZ-4-5-4PACK',
        ]);

        $bundle2->setBundleComponents([
            [
                'id' => $componentForBundle2->id,
                'quantity' => 1,
            ],
        ]);

        ProductListing::factory()->create([
            'product_id' => $bundle1->id,
            'document_id' => $shopifyProduct1->id,
            'document_type' => ShopifyProduct::class,
            'sales_channel_id' => 1,
            'sales_channel_listing_id' => '40934133563507',
        ]);
        ProductListing::factory()->create([
            'product_id' => $bundle2->id,
            'document_id' => $shopifyProduct2->id,
            'document_type' => ShopifyProduct::class,
            'sales_channel_id' => 1,
            'sales_channel_listing_id' => '39456218546291',
        ]);

        $orderPayload = file_get_contents(__DIR__.'/GML362172.json');
        $orderPayload = json_decode($orderPayload, 1);
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $integrationInstance->id,
            'order_number' => 'GML362172',
            'json_object' => $orderPayload,
        ]);
        $shopifyOrder->save();
        $salesOrder = $shopifyOrder->createSKUOrder();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML362172',
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $componentForBundle1->id,
            'bundle_id' => $bundle1->id,
        ]);

        $salesOrder->update([
            'ship_by_date' => now(),
            'deliver_by_date' => now(),
        ]);

        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $this->putJson(route('sales-orders.update', $salesOrder), [
            'sales_order_lines' => [
                [
                    'id' => $salesOrderLine->id,
                    'sales_channel_line_id' => '13397711224947',
                    'product_id' => $salesOrderLine->product_id,
                    'description' => $salesOrderLine->description,
                    'amount' => $salesOrderLine->amount,
                    'quantity' => $salesOrderLine->quantity,
                    'warehouse_id' => $newWarehouse->id,
                ],
            ],
        ])->assertSuccessful();

        $updatedOrderPayload = file_get_contents(__DIR__.'/GML362172-update.json');
        $updatedOrderPayload = json_decode($updatedOrderPayload, 1);
        $shopifyOrder->update(['json_object' => $updatedOrderPayload]);

        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 2);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_shopify_order_stays_reserved_after_update(): void
    {
        $integrationInstance = SalesChannel::where('id', 1)->first()->integrationInstance;
        $orderPayload = file_get_contents(__DIR__.'/GML362720.json');
        $orderPayload = json_decode($orderPayload, 1);
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $integrationInstance->id,
            'order_number' => 'GML362720',
            'json_object' => $orderPayload,
        ]);
        $shopifyOrder->save();
        $salesOrder = $shopifyOrder->createSKUOrder();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML362720',
            'order_status' => SalesOrder::STATUS_RESERVED,
        ]);

        app(SalesOrderManager::class)->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'ship_by_date' => now(),
                'deliver_by_date' => now(),
                'on_hold' => true,
            ]),
        ]));

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML362720',
            'order_status' => SalesOrder::STATUS_RESERVED,
        ]);

        $updatedOrderPayload = file_get_contents(__DIR__.'/GML362172-update.json');
        $updatedOrderPayload = json_decode($updatedOrderPayload, 1);
        $shopifyOrder->update(['json_object' => $updatedOrderPayload]);

        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML362720',
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_shopify_order_handles_canceled_quantity_complications(): void
    {

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::query()
            ->whereHas('integrationInstance', fn($q) => $q->where('name', Integration::NAME_SKU_IO))
            ->firstOrFail();
        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = $salesChannel->integrationInstance;

        $orderPayload = file_get_contents(__DIR__.'/GML374804.json');
        $orderPayload = json_decode($orderPayload, 1);
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $integrationInstance->id,
            'order_number' => 'GML374804',
            'json_object' => $orderPayload,
        ]);
        $shopifyOrder->save();

        $this->setUpProductsForCancelComplicationsTest($integrationInstance);

        $shopifyOrder->createSKUOrder();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML374804',
        ]);

        $shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_order_number' => 'GML374804',
        ]);

        $firstProduct = Product::where('sku', 'CR-CC-FZ-3OZ-2PACK')->first();
        $partiallyCanceledProduct = Product::where('sku', 'SC-FRESH-1LB-1')->first();

        // Originally CL-CC-FZ-1PT-1 appeared to be duplicating on updateSkuOrder but that doesn't seem to be happening anymore.
        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 9);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $firstProduct->id,
            'quantity' => 2,
            'canceled_quantity' => 0,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $partiallyCanceledProduct->id,
            'quantity' => 2,
            'canceled_quantity' => 6,
        ]);
    }

    /**
     * @throws ProductBundleException
     * @throws Throwable
     */
    private function setUpProductsForCancelComplicationsTest(IntegrationInstance $integrationInstance): void
    {
        $warehouse = Warehouse::first();

        $shopifyProductData = [
            [
                'variant_id' => 32108403294323,
                'title' => '2 Crab Cakes',
                'sku' => 'ADDON-2-CRAB-CAKES',
            ],
            [
                'variant_id' => 17426608783475,
                'title' => 'Fresh Maine Scallops - 2',
                'sku' => 'SHOPIFYHOLIDAYSCALLOPS2',
            ],
            [
                'variant_id' => 32210604392563,
                'title' => 'FRESHFISHSAMPLER',
                'sku' => 'FRESHFISHSAMPLER',
            ],
            [
                'variant_id' => 31895614619763,
                'title' => 'Gulf of Maine Bluefin Tuna - 2lbs',
                'sku' => 'SHOPIFYBLUEFINTUNA2LBS',
            ],
            [
                'variant_id' => 32106338713715,
                'title' => '1lb Tiger Shrimp',
                'sku' => 'ADDON-1-TIGER-SHRIMP',
            ],
            [
                'variant_id' => 32108399984755,
                'title' => '1 Pint Wicked Good Clam Chowda - Special Price',
                'sku' => 'ADDON-1-PINT-CLAM-CHOWDER',
            ],
        ];

        ShopifyProduct::factory(6)->state(new Sequence(
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[0]['variant_id']],
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[1]['variant_id']],
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[2]['variant_id']],
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[3]['variant_id']],
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[4]['variant_id']],
            ['integration_instance_id' => $integrationInstance->id, 'variant_id' => $shopifyProductData[5]['variant_id']],
        ))->create();

        for ($i = 0; $i < 6; $i++) {
            $shopifyProduct = ShopifyProduct::query()->offset($i)->first();
            $jsonObject = $shopifyProduct->json_object;
            $jsonObject['variant_id'] = $shopifyProductData[$i]['variant_id'];
            $jsonObject['title'] = $shopifyProductData[$i]['title'];
            $jsonObject['sku'] = $shopifyProductData[$i]['sku'];
            $shopifyProduct->json_object = $jsonObject;
            $shopifyProduct->save();
            $shopifyProduct->refresh();
        }

        $product1component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'CR-CC-FZ-3OZ-2PACK',
        ]);

        $product1component1->setInitialInventory($warehouse->id, 100, 5.00);

        $product1 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'ADDON-2-CRAB-CAKES',
        ]);

        $product1->setBundleComponents([
            [
                'id' => $product1component1->id,
                'quantity' => 1,
            ],
        ]);

        $product2component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'SC-FRESH-1LB-1',
        ]);

        $product2component1->setInitialInventory($warehouse->id, 100, 5.00);

        $product2 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'SHOPIFYHOLIDAYSCALLOPS2',
        ]);

        $product2->setBundleComponents([
            [
                'id' => $product2component1->id,
                'quantity' => 2,
            ],
        ]);

        $product3component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'HD-FRESH-1LB-1',
        ]);

        $product3component2 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'SM-FRESH-1LB-1',
        ]);

        $product3component3 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'SW-FRESH-1LB-1',
        ]);

        $product3component4 = $product2component1;

        $product3component1->setInitialInventory($warehouse->id, 100, 5.00);
        $product3component2->setInitialInventory($warehouse->id, 100, 5.00);
        $product3component3->setInitialInventory($warehouse->id, 100, 5.00);
        $product3component4->setInitialInventory($warehouse->id, 100, 5.00);

        $product3 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'FRESHFISHSAMPLER',
        ]);

        $product3->setBundleComponents([
            [
                'id' => $product3component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $product3component2->id,
                'quantity' => 1,
            ],
            [
                'id' => $product3component3->id,
                'quantity' => 1,
            ],
            [
                'id' => $product3component4->id,
                'quantity' => 1,
            ],
        ]);

        $product4component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'TN-BL-FRESH-1LB-1',
        ]);

        $product4component1->setInitialInventory($warehouse->id, 100, 5.00);

        $product4 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'SHOPIFYBLUEFINTUNA2LBS',
        ]);

        $product4->setBundleComponents([
            [
                'id' => $product4component1->id,
                'quantity' => 2,
            ],
        ]);

        $product5component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'SH-TS-FZ-1LB-1',
        ]);

        $product5component1->setInitialInventory($warehouse->id, 100, 5.00);

        $product5 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'ADDON-1-TIGER-SHRIMP',
        ]);

        $product5->setBundleComponents([
            [
                'id' => $product5component1->id,
                'quantity' => 1,
            ],
        ]);

        $product6component1 = Product::factory()->create([
            'type' => Product::TYPE_STANDARD,
            'sku' => 'CL-CC-FZ-1PT-1',
        ]);

        $product6component1->setInitialInventory($warehouse->id, 100, 5.00);

        $product6 = Product::factory()->create([
            'type' => Product::TYPE_BUNDLE,
            'sku' => 'ADDON-1-PINT-CLAM-CHOWDER',
        ]);

        $product6->setBundleComponents([
            [
                'id' => $product6component1->id,
                'quantity' => 1,
            ],
        ]);

        for ($i = 0; $i < 6; $i++) {
            $shopifyProduct = ShopifyProduct::query()->offset($i)->first();
            ProductListing::factory()->create([
                'sales_channel_id' => $integrationInstance->salesChannel->id,
                'sales_channel_listing_id' => $shopifyProduct->variant_id,
                'listing_sku' => $shopifyProduct->sku,
                'product_id' => Product::query()->where('sku', $shopifyProduct->sku)->first()->id,
                'document_type' => ShopifyProduct::class,
                'document_id' => $shopifyProduct->id
            ]);
        }
    }
}
