<?php

namespace Tests\Unit\ShopifyOrders;

use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\FinancialLine;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderMapping;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\SettingRepository;
use App\Repositories\WarehouseRepository;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class CreateShopifyOrderTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    /**
     *
     *  Test edge cases such as:
     *  Shipping lines when tax is included (SKU-6472)
     *
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_it_can_create_sku_order_from_shopify(): void
    {
        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => '323',
            'json_object' => ShopifyTestingData::getOrderWithTaxInclusiveShipping()
        ]);
        $shopifyOrder->createSKUOrder();
        $this->assertDatabaseHas((new FinancialLine())->getTable(), [
            'amount' => 9.90,
            'tax_allocation' => 0.90,
        ]);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_it_wont_fulfill_if_not_single_shippable_warehouse(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        Warehouse::factory(2)->create([
            'type' => 'direct',
        ]);

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'partial',
                'financial_status' => 'paid',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                        'is_product' => true,
                    ],
                ],
                'refunds' => [],
                'fulfillments' => [
                    [
                        'id' => 1,
                        'status' => 'success',
                        'tracking_company' => 'USPS',
                        'tracking_numbers' => ['1Z2345'],
                        'line_items' => [
                            [
                                'id' => 1,
                                'quantity' => 2,
                                'sku' => 'SKU-1',
                                'variant_id' => 1,
                                'product_exists' => true,
                            ],
                        ],
                        'created_at' => now()->subDay(),
                        'updated_at' => now()->subDay(),
                    ],
                ],
            ],
        ]);
        $shopifyOrder->save();

        /** @var SalesOrder $salesOrder */
        $salesOrder = $shopifyOrder->createSKUOrder();

        Sanctum::actingAs(User::factory()->create());

        $this->getJson('/api/sales-orders/'.$salesOrder->id.'/compare-fulfillments-with-sales-channel')
            ->assertOk()
            ->assertJsonCount(1, 'data.fulfillmentToCreate');

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 1);
        // No fulfillments should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 5,
            'externally_fulfilled_quantity' => 0,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);
    }

    /**
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     * @throws BindingResolutionException
     */
    public function test_it_will_fulfill_if_single_shippable_warehouse(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        $this->assertTrue(app(WarehouseRepository::class)->hasSingleShippableWarehouse());

        /** @var Product $product */
        $product = Product::factory()->create([
            'sku' => 'SKU-1',
        ]);

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'product_id' => $product->id,
            'price' => 10,
        ]);

        // TODO: Still need two tests for automated fulfillment
        // 1. if the fulfillment was already created, it should not fulfill again since there is nothing to fulfill
        // 2. If the fulfillment was not created, it should fulfill the order (if it qualifies)
        //        $integration = Integration::query()->whereName(Integration::NAME_STARSHIPIT)->first();
        //
        //        // Create the Starshipit integration instance.
        //        IntegrationInstance::factory()->create([
        //            'name' => Integration::NAME_STARSHIPIT,
        //            'integration_id' => $integration ? $integration->id : Integration::factory()->create([
        //                'name' => Integration::NAME_STARSHIPIT,
        //                'integration_type' => Integration::TYPE_SHIPPING_PROVIDER
        //            ]),
        //            'integration_settings' => [
        //                'fulfillment' => [
        //                    'automatedWarehousesIds' => [
        //                        app(WarehouseRepository::class)->getSingleShippableWarehouse()->id
        //                    ]
        //                ]
        //            ]
        //        ]);

        Sanctum::actingAs(User::factory()->create());

        $this->postJson(route('inventory-adjustments.store'), [
            'product_id' => $product->id,
            'adjustment_date' => now()->format('Y-m-d'),
            'quantity' => 10,
            'warehouse_id' => app(WarehouseRepository::class)->getSingleShippableWarehouse()->id,
            'reason' => 'test',
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'partial',
                'financial_status' => 'paid',
                'line_items' => [
                    [
                        'id' => 1234,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'price' => 20.00,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                        'tax_lines' => [
                            [
                                'price' => 10.00,
                                'rate' => 0.10,
                                'title' => 'GST',
                            ],
                        ],
                    ],
                    [
                        'id' => '3434',
                        'gift_card' => true,
                        'name' => 'Gift Card 123',
                        'quantity' => 1,
                        'price' => 4.00,
                        'tax_lines' => [
                            [
                                'price' => 0.20,
                                'rate' => 0.05,
                                'title' => 'PST',
                            ],
                        ],

                    ],
                ],
                'refunds' => [],
                'fulfillments' => [
                    [
                        'id' => 1,
                        'status' => 'success',
                        'service' => 'manual',
                        'tracking_company' => 'USPS',
                        'tracking_numbers' => ['1Z2345'],
                        'line_items' => [
                            [
                                'id' => 1234,
                                'quantity' => 2,
                                'sku' => 'SKU-1',
                                'variant_id' => 1,
                                'product_exists' => true,
                            ],
                        ],
                        'created_at' => now()->subDay(),
                        'updated_at' => now()->subDay(),
                    ],
                ],
                'shipping_lines' => [
                    [
                        'price' => 20.00,
                        'code' => 'Free Shipping',
                        'total_tax' => 2.00,
                        'tax_lines' => [
                            [
                                'rate' => 0.1,
                                'title' => 'GST',
                                'price' => 2.00,
                            ],
                        ],
                    ],
                ],
                'tags' => 'tag1, tag2',
            ],
        ]);
        $shopifyOrder->save();

        $salesOrder = $shopifyOrder->createSKUOrder();

        $this->assertEquals(
            'tag1',
            $salesOrder->tags()->first()->name
        );

        $this->assertEquals(
            'tag2',
            $salesOrder->tags()->offset(1)->first()->name
        );

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 1);
        // No fulfillments should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 5,
            'externally_fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseHas((new ShopifyOrderMapping())->getTable(), [
            'shopify_order_id' => $shopifyOrder->id,
            'line_id' => 1234,
            'line_type' => 'line_item',
            'quantity' => 2,
            'link_id' => 1,
            'link_type' => 'fulfillments',
            'sku_link_type' => SalesOrderFulfillment::class,
        ]);

        /*
         * 1 for the initial inventory adjustment
         * 2 inventory movements for reservation
         * 1 for reservation release
         */
        $this->assertDatabaseCount('inventory_movements', 4);
    }

    public function test_it_will_split_externally_fulfilled_quantity_lines(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();

        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 1,
            'price' => 10,
        ]);
        ProductListing::factory()->create([
            'sales_channel_id' => $salesChannel->id,
            'sales_channel_listing_id' => 2,
            'price' => 15,
        ]);

        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'SHOPIFY-ORDER-NAME',
            'json_object' => [
                'name' => 'SHOPIFY-ORDER-NAME',
                'fulfillment_status' => 'partial',
                'financial_status' => 'paid',
                'line_items' => [
                    [
                        'id' => 1,
                        'sku' => 'SKU-1',
                        'product_exists' => true,
                        'quantity' => 5,
                        'name' => $this->faker->sentence(),
                        'variant_id' => 1,
                        'is_product' => true,
                    ],
                ],
                'refunds' => [],
                'fulfillments' => [
                    [
                        'id' => 1,
                        'status' => 'success',
                        'tracking_company' => 'USPS',
                        'tracking_numbers' => ['1Z2345'],
                        'line_items' => [
                            [
                                'id' => 1,
                                'quantity' => 2,
                                'sku' => 'SKU-1',
                                'variant_id' => 1,
                                'product_exists' => true,
                            ],
                        ],
                        'created_at' => SettingRepository::getInventoryStartDate()->subDay(),
                        'updated_at' => now()->subDay(),
                    ],
                ],
            ],
        ]);
        $shopifyOrder->save();

        $salesOrder = $shopifyOrder->createSKUOrder();

        $this->assertDatabaseCount('sales_orders', 1);
        $this->assertDatabaseCount('sales_order_lines', 2);
        // No fulfillments should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => $shopifyOrder->name,
            'order_status' => SalesOrder::STATUS_OPEN,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 3,
            'externally_fulfilled_quantity' => 0,
        ]);
        $this->assertDatabaseHas('sales_order_lines', [
            'quantity' => 2,
            'externally_fulfilled_quantity' => 2,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);

        app(SalesOrderRepository::class)->bulkDelete(Arr::wrap($salesOrder->id));

        $this->assertDatabaseCount('sales_orders', 0);
    }

    public function test_refund_link_generation(): void
    {
        Queue::fake();

        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integration_instance_id,
            'order_number' => 'C402972',
            'json_object' => [
                'id' => 4840898854985,
                'currency' => 'AUD',
                'fulfillment_status' => null,
                'name' => 'C402972',
                'note' => null,
                'number' => 401972,
                'order_number' => 402972,
                'total_price' => '382.52',
                'fulfillments' => [
                ],
                'line_items' => [
                    [
                        'id' => 12204275236937,
                        'name' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Cafe - 24F',
                        'price' => '74.95',
                        'product_exists' => true,
                        'product_id' => 6563915137097,
                        'discount_allocations' => [
                        ],
                        'tax_lines' => [
                        ],
                        'quantity' => 1,
                        'sku' => '840468115690',
                        'title' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Cafe',
                        'variant_id' => 39290632273993,

                    ],
                    [
                        'id' => 12204275269705,
                        'name' => 'Glamorise MagicLift Cotton Support Wire-free Bra - Black - 24F',
                        'price' => '84.95',
                        'product_exists' => true,
                        'product_id' => 6890937974857,
                        'quantity' => 1,
                        'sku' => '889902106092',
                        'title' => 'Glamorise MagicLift Cotton Support Wire-free Bra - Black',
                        'variant_id' => 40481677181001,
                        'tax_lines' => [],
                        'discount_allocations' => [
                        ],
                    ],
                    [
                        'id' => 12204275302473,
                        'name' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Soft Gray - 24F',
                        'price' => '74.95',
                        'product_exists' => true,
                        'product_id' => 6563370270793,
                        'quantity' => 1,
                        'requires_shipping' => true,
                        'sku' => '889902002189',
                        'taxable' => true,
                        'title' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Soft Gray',
                        'variant_id' => 39289908559945,
                        'tax_lines' => [],
                        'discount_allocations' => [
                        ],
                    ],
                    [
                        'id' => 12204275335241,
                        'admin_graphql_api_id' => 'gid://shopify/LineItem/12204275335241',
                        'fulfillable_quantity' => 1,
                        'fulfillment_service' => 'manual',
                        'name' => 'Premium Shipping Protection - 5.25',
                        'price' => '5.25',
                        'product_exists' => true,
                        'product_id' => 6890154131529,
                        'quantity' => 1,
                        'sku' => 'NVDPROTECTION7',
                        'title' => 'Premium Shipping Protection',
                        'variant_id' => 40475366359113,
                        'variant_inventory_management' => 'shopify',
                        'variant_title' => '5.25',
                        'vendor' => 'Curvy',
                        'tax_lines' => [
                        ],

                        'discount_allocations' => [
                        ],
                    ],
                    [
                        'id' => 12204276318281,
                        'fulfillable_quantity' => 0,
                        'fulfillment_service' => 'manual',
                        'fulfillment_status' => null,
                        'gift_card' => false,
                        'grams' => 120,
                        'name' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Cafe - 24F',
                        'price' => '74.95',
                        'product_exists' => true,
                        'product_id' => 6563915137097,
                        'quantity' => 1,
                        'requires_shipping' => true,
                        'sku' => '840468115690',
                        'title' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Cafe',
                        'variant_id' => 39290632273993,
                        'tax_lines' => [],
                        'discount_allocations' => [

                        ],
                    ],
                    [
                        'id' => 12204276711497,
                        'price' => '74.95',
                        'product_exists' => true,
                        'product_id' => 6563915137097,
                        'quantity' => 1,
                        'sku' => '840468115690',
                        'taxable' => true,
                        'name' => 'Glamorise No-Bounce Camisole Wire-free Sports Bra - Cafe',
                        'variant_id' => 39290632273993,
                        'tax_lines' => [],
                        'discount_allocations' => [

                        ],
                    ],
                ],
                'refunds' => [
                    [
                        'id' => 875329978441,
                        'note' => '',
                        'order_id' => 4840898854985,
                        'processed_at' => '2023-08-03T13:10:54+10:00',
                        'restock' => true,
                        'user_id' => 41427632201,
                        'order_adjustments' => [
                        ],
                        'refund_line_items' => [
                            [
                                'id' => 391254638665,
                                'line_item_id' => 12204276318281,
                                'location_id' => 16349331529,
                                'quantity' => 1,
                                'restock_type' => 'cancel',
                                'subtotal' => 71.21,
                                'total_tax' => 6.47,

                            ],
                            [
                                'id' => 391254671433,
                                'line_item_id' => 12204276711497,
                                'location_id' => 16349331529,
                                'quantity' => 1,
                                'restock_type' => 'cancel',
                                'subtotal' => 71.21,
                                'total_tax' => 6.47,

                            ],
                        ],

                    ],
                    [
                        'id' => 875330011209,
                        'note' => 'DISCOUNT',
                        'order_id' => 4840898854985,
                        'processed_at' => '2023-08-03T13:11:21+10:00',
                        'restock' => false,
                        'user_id' => 41427632201,
                        'refund_line_items' => [],
                        'order_adjustments' => [
                            [
                                'id' => 217043632201,
                                'amount' => '-24.10',
                                'kind' => 'refund_discrepancy',
                                'order_id' => 4840898854985,
                                'reason' => 'Refund discrepancy',
                                'refund_id' => 875330011209,
                                'tax_amount' => 0,
                            ],
                        ],

                    ],
                ],

            ],
        ]);
        $shopifyOrder->save();
        $shopifyOrder->createSKUOrder();
        $this->assertDatabaseCount('sales_credit_lines', 3);

        $this->assertDatabaseHas('shopify_order_mappings', [
            'sku_link_type' => SalesCreditLine::class,
            'line_type' => 'line_item',
        ]);
        $this->assertDatabaseHas('shopify_order_mappings', [
            'sku_link_type' => SalesCreditLine::class,
        ]);
        $this->assertDatabaseHas('shopify_order_mappings', [
            'sku_link_type' => SalesCreditLine::class,
        ]);
    }

    public function test_shopify_orders_can_shipping_line_discounts(): void
    {
        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create();
        $shopifyOrder = new ShopifyOrder([
            'integration_instance_id' => $salesChannel->integrationInstance->id,
            'order_number' => '17459',
            'json_object' => [
                "id" => 5460905328880,
                "cancelled_at" => null,
                "closed_at" => null,
                "confirmed" => true,
                "contact_email" => "betsimorrison@gmail.com",
                "created_at" => "2023-12-12T10:39:30-08:00",
                "currency" => "USD",
                "current_subtotal_price" => "271.35",
                "current_total_discounts" => "59.88",
                "current_total_price" => "271.35",
                "current_total_tax" => "0.00",
                "email" => "betsimorrison@gmail.com",
                "financial_status" => "paid",
                "fulfillment_status" => null,
                "location_id" => null,
                "name" => "#17459",
                "note" => null,
                "note_attributes" => [],
                "number" => 16459,
                "order_number" => 17459,
                "payment_gateway_names" => ["shopify_payments"],
                "phone" => null,
                "presentment_currency" => "USD",
                "processed_at" => "2023-12-12T10:39:26-08:00",
                "subtotal_price" => "271.35",
                "tags" => "",
                "tax_lines" => [],
                "taxes_included" => false,
                "total_discounts" => "59.88",
                "total_line_items_price" => "319.23",
                "total_outstanding" => "0.00",
                "total_price" => "271.35",
                "total_tax" => "0.00",
                "updated_at" => "2023-12-12T10:39:33-08:00",
                "billing_address" => [
                    "first_name" => "Betsi",
                    "address1" => "173 Vista Drive",
                    "phone" => "+14062708799",
                    "city" => "Whitefish",
                    "zip" => "59937",
                    "province" => "Montana",
                    "country" => "United States",
                    "last_name" => "Morrison",
                    "address2" => null,
                    "company" => null,
                    "name" => "Betsi Morrison",
                    "country_code" => "US",
                    "province_code" => "MT"
                ],
                "customer" => [
                    "id" => 7067682701552,
                    "email" => "betsimorrison@gmail.com",
                    "created_at" => "2023-12-12T10:39:26-08:00",
                    "updated_at" => "2023-12-12T10:51:25-08:00",
                    "first_name" => "Betsi",
                    "last_name" => "Morrison",
                    "state" => "disabled",
                    "phone" => null,
                    "default_address" => [
                        "id" => 8655354822896,
                        "customer_id" => 7067682701552,
                        "first_name" => "Betsi",
                        "last_name" => "Morrison",
                        "company" => null,
                        "address1" => "173 Vista Drive",
                        "address2" => null,
                        "city" => "Whitefish",
                        "province" => "Montana",
                        "country" => "United States",
                        "zip" => "59937",
                        "phone" => "+14062708799",
                        "name" => "Betsi Morrison",
                        "province_code" => "MT",
                        "country_code" => "US",
                        "country_name" => "United States",
                        "default" => true
                    ]
                ],
                "discount_applications" => [
                    [
                        "target_type" => "line_item",
                        "type" => "automatic",
                        "value" => "15.0",
                        "value_type" => "percentage",
                        "allocation_method" => "across",
                        "target_selection" => "entitled",
                        "title" => "December Weekly Surprise Deal: Week 2 Exclusive"
                    ],
                    [
                        "target_type" => "shipping_line",
                        "type" => "automatic",
                        "value" => "100.0",
                        "value_type" => "percentage",
                        "allocation_method" => "each",
                        "target_selection" => "entitled",
                        "title" => "Free Shipping December"
                    ]
                ],
                "fulfillments" => [],
                "line_items" => [
                    [
                        "id" => 13549728661744,
                        "fulfillable_quantity" => 9,
                        "fulfillment_service" => "manual",
                        "fulfillment_status" => null,
                        "gift_card" => false,
                        "grams" => 0,
                        "name" => "32.8ft LED Strip Lights, HitLights 5050 RGB Color Changing",
                        "price" => "35.47",
                        "product_exists" => true,
                        "product_id" => 4776931917903,
                        "properties" => [],
                        "quantity" => 9,
                        "requires_shipping" => true,
                        "sku" => "L1012V-MC1-3230-k",
                        "taxable" => true,
                        "title" => "32.8ft LED Strip Lights, HitLights 5050 RGB Color Changing",
                        "total_discount" => "0.00",
                        "variant_id" => 42781145825520,
                        "variant_inventory_management" => "shopify",
                        "variant_title" => null,
                        "vendor" => "HitLights Amazon",
                        "tax_lines" => [],
                        "duties" => [],
                        "discount_allocations" => [
                            [
                                "amount" => "47.88",
                            ]
                        ]
                    ]
                ],
                "refunds" => [],
                "shipping_address" => [
                    "first_name" => "Betsi",
                    "address1" => "173 Vista Drive",
                    "phone" => "+14062708799",
                    "city" => "Whitefish",
                    "zip" => "59937",
                    "province" => "Montana",
                    "country" => "United States",
                    "last_name" => "Morrison",
                    "address2" => null,
                    "company" => null,
                    "name" => "Betsi Morrison",
                    "country_code" => "US",
                    "province_code" => "MT"
                ],
                "shipping_lines" => [
                    [
                        "id" => 4443111293168,
                        "carrier_identifier" => "650f1a14fa979ec5c74d063e968411d4",
                        "code" => "2 Day Shipping - Flat Rate - Cutoff Time: 2PM PST",
                        "delivery_category" => null,
                        "discounted_price" => "12.00",
                        "phone" => null,
                        "price" => "12.00",
                        "requested_fulfillment_service_id" => null,
                        "source" => "shopify",
                        "title" => "2 Day Shipping - Flat Rate - Cutoff Time: 2PM PST",
                        "tax_lines" => [],
                        "discount_allocations" => [
                            [
                                "amount" => "12.00",
                                "amount_set" => [
                                    "shop_money" => [
                                        "amount" => "12.00",
                                        "currency_code" => "USD"
                                    ],
                                    "presentment_money" => [
                                        "amount" => "12.00",
                                        "currency_code" => "USD"
                                    ]
                                ],
                                "discount_application_index" => 1
                            ]
                        ]
                    ]
                ]
            ]
        ]);
        $shopifyOrder->save();

        $this->assertEmpty($shopifyOrder->getSkuFinancialLines());
    }
}
