<?php

namespace Tests\Unit;

use App\Exceptions\ProductBundleException;
use App\Managers\ReportManager;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Warehouse;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ReportSalesChannelCoverageTest extends TestCase
{
    use FastRefreshDatabase;

    private ReportManager $reporting;

    public function setUp(): void
    {
        parent::setUp();

        $this->reporting = app(ReportManager::class);
    }

    /**
     * @throws Throwable
     */
    public function test_it_does_not_consider_inactive_listings_as_coverage(): void
    {
        $ebayIntegration = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'Ebay'
            ]);

        $amazonIntegration = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'Amazon'
            ]);

        $wooCommerceIntegration = WooCommerceIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'WooCommerce'
            ]);

        $shopifyIntegration  = IntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'Shopify',
                'integration_id' => Integration::where('name', Integration::NAME_SHOPIFY)->first()->id
            ]);

        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'sku' => 'ABC',
            'name' => 'Product Name',
            'type' => Product::TYPE_STANDARD,
        ]);

        $product->setInitialInventory($warehouse->id, 10, 5.00);

        $ebayProduct = $this->setupEbayProduct($ebayIntegration, $product, 'Active');
        $wooCommerceProduct = $this->setupWooCommerceProduct($wooCommerceIntegration, $product, 'publish');
        $amazonProduct = $this->setupAmazonProduct($amazonIntegration, $product, 'Active');
        $shopifyProduct = $this->setupShopifyProduct($shopifyIntegration, $product, 'active');
        $integrationIds = [
            $ebayIntegration->id,
            $wooCommerceIntegration->id,
            $amazonIntegration->id,
            $shopifyIntegration->id,
        ];

        $coverages = $this->reporting->getSalesChannelCoverage(false, false, $integrationIds);

        $this->assertEquals($coverages, [
            [
                'id' => $product->id,
                'sku' => 'ABC',
                'supplier' => [
                    'id' => null,
                    'name' => null,
                ],
                'type' => Product::TYPE_STANDARD,
                'stock' => 10,
                'valuation' => 50.00,
                'coverage' => [
                    'Ebay' => 1,
                    'WooCommerce' => 1,
                    'Amazon' => 1,
                    'Shopify' => 1,
                    'Total' => 4,
                ]
            ],
        ]);

        $jsonObject = $ebayProduct->json_object;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Completed';
        $ebayProduct->json_object = $jsonObject;
        $ebayProduct->save();
        $ebayProduct->refresh();

        $jsonObject = $wooCommerceProduct->json_object;
        $jsonObject['status'] = 'draft';
        $wooCommerceProduct->json_object = $jsonObject;
        $wooCommerceProduct->save();
        $wooCommerceProduct->refresh();

        $jsonObject = $amazonProduct->json_object;
        $jsonObject['status'] = 'Inactive';
        $amazonProduct->json_object = $jsonObject;
        $amazonProduct->save();
        $amazonProduct->refresh();

        $jsonObject = $shopifyProduct->json_object;
        $jsonObject['status'] = 'draft';
        $shopifyProduct->json_object = $jsonObject;
        $shopifyProduct->save();
        $shopifyProduct->refresh();

        $coverages = $this->reporting->getSalesChannelCoverage(false, false, $integrationIds);

        $this->assertEquals($coverages, [
            [
                'id' => $product->id,
                'sku' => 'ABC',
                'supplier' => [
                    'id' => null,
                    'name' => null,
                ],
                'type' => Product::TYPE_STANDARD,
                'stock' => 10,
                'valuation' => 50.00,
                'coverage' => [
                    'Ebay' => 0,
                    'WooCommerce' => 0,
                    'Amazon' => 0,
                    'Shopify' => 0,
                    'Total' => 0,
                ]
            ],
        ]);
    }

    /**
     * @throws ProductBundleException
     * @throws Throwable
     */
    public function test_it_considers_listings_for_parents_of_components_as_coverage(): void
    {
        $ebayIntegration = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'name' => 'Ebay'
            ]);

        $warehouse = Warehouse::first();

        $kit = Product::factory()->create([
            'sku' => 'ABC',
            'name' => 'Product Name',
            'type' => Product::TYPE_KIT,
        ]);

        $component = Product::factory()->create([
            'sku' => 'DEF',
            'name' => 'Component Name',
            'type' => Product::TYPE_STANDARD,
        ]);

        $component->setInitialInventory($warehouse->id, 10, 5.00);

        $this->setupEbayProduct($ebayIntegration, $kit, 'Active');

        $coverages = $this->reporting->getSalesChannelCoverage(false, false, [$ebayIntegration->id]);

        $this->assertEquals($coverages, [
            [
                'id' => $component->id,
                'sku' => 'DEF',
                'supplier' => [
                    'id' => null,
                    'name' => null,
                ],
                'type' => Product::TYPE_STANDARD,
                'stock' => 10,
                'valuation' => 50.00,
                'coverage' => [
                    'Total' => 0,
                ]
            ],
        ]);

        $kit->setBundleComponents([
            [
                'id' => $component->id,
                'quantity' => 2,
            ],
        ]);

        $coverages = $this->reporting->getSalesChannelCoverage(false, false, [$ebayIntegration->id]);

        $this->assertEquals($coverages, [
            [
                'id' => $component->id,
                'sku' => 'DEF',
                'supplier' => [
                    'id' => null,
                    'name' => null,
                ],
                'type' => Product::TYPE_STANDARD,
                'stock' => 10,
                'valuation' => 50.00,
                'coverage' => [
                    'Ebay' => 1,
                    'Total' => 1,
                ]
            ],
        ]);

    }

    private function setupEbayProduct(EbayIntegrationInstance $ebayIntegration, Product $product, string $status): EbayLegacyProduct
    {
        $ebayProduct = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $ebayIntegration->id,
        ]);

        $jsonObject = $ebayProduct->json_object;
        $jsonObject['SellingStatus']['ListingStatus'] = $status;
        $ebayProduct->json_object = $jsonObject;
        $ebayProduct->save();
        $ebayProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $ebayIntegration->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => EbayLegacyProduct::class,
            'document_id' => $ebayProduct->id,
        ]);

        return $ebayProduct;
    }

    private function setupWooCommerceProduct(WooCommerceIntegrationInstance $wooCommerceIntegration, Product $product, string $status): WooCommerceProduct
    {
        $wooCommerceProduct = WooCommerceProduct::factory()->create([
            'integration_instance_id' => $wooCommerceIntegration->id,
        ]);

        $jsonObject = $wooCommerceProduct->json_object;
        $jsonObject['status'] = $status;
        $wooCommerceProduct->json_object = $jsonObject;
        $wooCommerceProduct->save();
        $wooCommerceProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $wooCommerceIntegration->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => WooCommerceProduct::class,
            'document_id' => $wooCommerceProduct->id,
        ]);

        return $wooCommerceProduct;
    }

    private function setupAmazonProduct(AmazonIntegrationInstance $amazonIntegration, Product $product, string $status): AmazonProduct
    {
        $amazonProduct = AmazonProduct::factory()->create([
            'integration_instance_id' => $amazonIntegration->id,
        ]);

        $jsonObject = $amazonProduct->json_object;
        $jsonObject['status'] = $status;
        $amazonProduct->json_object = $jsonObject;
        $amazonProduct->save();
        $amazonProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $amazonIntegration->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => AmazonProduct::class,
            'document_id' => $amazonProduct->id,
        ]);

        return $amazonProduct;
    }

    private function setupShopifyProduct(IntegrationInstance $shopifyIntegration, Product $product, string $status): ShopifyProduct
    {
        $shopifyProduct = ShopifyProduct::factory()->create([
            'integration_instance_id' => $shopifyIntegration->id,
            'removed_from_shopify' => false,
        ]);

        $jsonObject = $shopifyProduct->json_object;
        $jsonObject['status'] = $status;
        $shopifyProduct->json_object = $jsonObject;
        $shopifyProduct->save();
        $shopifyProduct->refresh();

        ProductListing::factory()->create([
            'sales_channel_id' => $shopifyIntegration->salesChannel->id,
            'product_id' => $product->id,
            'document_type' => ShopifyProduct::class,
            'document_id' => $shopifyProduct->id,
        ]);

        return $shopifyProduct;
    }
}
