<?php

namespace Tests\Unit;

use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\Supplier;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class PurchaseOrderManagerTest extends TestCase
{

    use FastRefreshDatabase;

    public function test_it_can_check_changing_lines_when_product_order_lines_do_not_exist(): void
    {
        $supplier = new Supplier();
        $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);
        $order = new PurchaseOrder();

        $this->assertFalse(
            $purchaseOrderManager->changingProductLines(
                purchaseOrder: $order,
                payload: []
            )
        );
    }

    public function test_it_can_check_changing_lines_when_non_product_purchase_order_lines_exist(): void
    {
        $supplier = new Supplier();
        $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);
        $order = new PurchaseOrder();

        $this->assertFalse(
            $purchaseOrderManager->changingProductLines(
                purchaseOrder: $order,
                payload: [
                    'purchase_order_lines' => [
                        [
                            'quantity' => 1,
                            'amount' => 1,
                        ]
                    ]
                ]
            )
        );
    }

    public function test_it_can_check_changing_lines_when_product_order_lines_with_product_id_exist(): void{
        $supplier = new Supplier();
        $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);
        $order = new PurchaseOrder();

        /** @var Product $product */
        $product = Product::factory()->create();

        $this->assertTrue(
            $purchaseOrderManager->changingProductLines(
                purchaseOrder: $order,
                payload: [
                    'purchase_order_lines' => [
                        [
                            'product_id' => $product->id,
                            'quantity' => 1,
                            'amount' => 1,
                        ]
                    ]
                ]
            )
        );
    }

    public function test_it_can_detect_deleting_product_purchase_order_line(): void{

        $supplier = Supplier::factory()->create();
        $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);

        /** @var PurchaseOrder $order */
        $order = PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory()->count(2))
            ->create();

        $this->assertTrue(
            $purchaseOrderManager->changingProductLines(
                purchaseOrder: $order,
                payload: [
                    'purchase_order_lines' => $order->purchaseOrderLines()->first()->toArray()
                ]
            )
        );
    }

    public function test_it_can_detect_deleting_all_product_purchase_order_lines(): void{

            $supplier = Supplier::factory()->create();
            $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);

            /** @var PurchaseOrder $order */
            $order = PurchaseOrder::factory()
                ->has(PurchaseOrderLine::factory()->count(2))
                ->create();

            $this->assertTrue(
                $purchaseOrderManager->changingProductLines(
                    purchaseOrder: $order,
                    payload: [
                        'purchase_order_lines' => []
                    ]
                )
            );
    }


    public function test_it_can_detect_changing_product_lines_when_new_product_line_is_added(): void{

        $supplier = Supplier::factory()->create();
        $purchaseOrderManager = new PurchaseOrderManager(supplier: $supplier);

        /** @var PurchaseOrder $order */
        $order = PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory()->count(2))
            ->create();

        /** @var Product $product */
        $product = Product::factory()->create();

        $this->assertTrue(
            $purchaseOrderManager->changingProductLines(
                purchaseOrder: $order,
                payload: [
                    'purchase_order_lines' => [
                        [
                            'product_id' => $product->id,
                            'quantity' => 1,
                            'amount' => 1,
                        ]
                    ]
                ]
            )
        );
    }
}
