<?php

namespace Tests\Unit;

use App\Console\Commands\Inventory\Integrity\MismatchedSalesOrderLineAndMovementQuantities;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

/**
 *
 * @group manual
 */
class MismatchedSalesOrderLineAndMovementQuantitiesTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     */
    public function test_it_can_identify_and_remedy_fulfilled_sales_order_lines_without_movements()
    {
        $mismatch = app(MismatchedSalesOrderLineAndMovementQuantities::class);

        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 1,
            ])
            ->create();

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 1,
                ]
            ]
        ], false, false);

        $salesOrderFulfillmentLine = SalesOrderFulfillmentLine::first();

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_type' => SalesOrderFulfillmentLine::class,
            'link_id' => $salesOrderFulfillmentLine->id,
        ]);

        $fulfillmentMovement = InventoryMovement::where('link_type', SalesOrderFulfillmentLine::class)
            ->where('link_id', $salesOrderFulfillmentLine->id)
            ->first();

        $this->assertEmpty($mismatch->fulfillmentLinesWithoutMovementsQuery()->count());

        $fulfillmentMovement->delete();

        $this->assertEquals(1, $mismatch->fulfillmentLinesWithoutMovementsQuery()->count());

        $mismatch->fixFulfillmentLinesWithoutMovements();

        $this->assertEmpty($mismatch->fulfillmentLinesWithoutMovementsQuery()->count());
    }
}
