<?php

namespace Tests\Unit;
use App\Integrations\Http\IntegrationHttpClient;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use Illuminate\Support\Facades\Http;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use RuntimeException;
use Tests\TestCase;

class IntegrationHttpClientTest extends TestCase
{

    use FastRefreshDatabase;

    private IntegrationHttpClient $client;

    public function setUp(): void
    {
        parent::setUp();

        /** @var IntegrationInstance $instance */
        $instance = IntegrationInstance::query()->where('name', Integration::NAME_SKU_IO)->firstOrFail();
        $this->client = new IntegrationHttpClient($instance);

        Http::preventStrayRequests();
    }

    public function test_it_can_perform_a_post_request()
    {
        Http::fake([
            '*' => Http::response(['status' => 'success']),
        ]);

        $response = $this->client->post('https://example.com', ['foo' => 'bar']);
        $this->assertEquals(200, $response->status());

        Http::assertSentCount(1);
    }

    public function test_it_can_perform_a_get_request()
    {
        Http::fake([
            '*' => Http::response(['status' => 'success']),
        ]);

        $response = $this->client->get('https://example.com');
        $this->assertEquals(200, $response->status());

        Http::assertSentCount(1);
    }

    public function test_it_can_perform_a_put_request()
    {
        Http::fake([
            '*' => Http::response(['status' => 'success']),
        ]);

        $response = $this->client->put('https://example.com', ['foo' => 'bar']);
        $this->assertEquals(200, $response->status());

        Http::assertSentCount(1);
    }

    public function test_it_can_perform_a_patch_request()
    {
        Http::fake([
            '*' => Http::response(['status' => 'success']),
        ]);

        $response = $this->client->patch('https://example.com', ['foo' => 'bar']);
        $this->assertEquals(200, $response->status());

        Http::assertSentCount(1);
    }

    public function test_it_can_perform_a_delete_request()
    {
        Http::fake([
            '*' => Http::response(['status' => 'success']),
        ]);

        $response = $this->client->delete('https://example.com');
        $this->assertEquals(200, $response->status());

        Http::assertSentCount(1);
    }

    public function test_it_can_retry_failed_get_request(): void{

        $this->expectException(RuntimeException::class);

        Http::fake([
            '*' => Http::response(['status' => 'error'], 500),
        ]);

        $response = $this->client->get('https://example.com', tries: 3);
        $this->assertEquals(500, $response->status());

        Http::assertSentCount(3);
    }

    public function test_it_can_retry_failed_post_request(): void{

        $this->expectException(RuntimeException::class);

        Http::fake([
            '*' => Http::response(['status' => 'error'], 500),
        ]);

        $response = $this->client->post('https://example.com', ['foo' => 'bar'], tries: 3);
        $this->assertEquals(500, $response->status());

        Http::assertSentCount(3);
    }

    public function test_it_can_retry_failed_put_request(): void{

        $this->expectException(RuntimeException::class);

        Http::fake([
            '*' => Http::response(['status' => 'error'], 500),
        ]);

        $response = $this->client->put('https://example.com', ['foo' => 'bar'], tries: 3);
        $this->assertEquals(500, $response->status());

        Http::assertSentCount(3);
    }

    public function test_it_can_retry_failed_patch_request(): void{

        $this->expectException(RuntimeException::class);

        Http::fake([
            '*' => Http::response(['status' => 'error'], 500),
        ]);

        $response = $this->client->patch('https://example.com', ['foo' => 'bar'], tries: 3);
        $this->assertEquals(500, $response->status());

        Http::assertSentCount(3);
    }

    public function test_it_can_retry_failed_delete_request(): void{

        $this->expectException(RuntimeException::class);

        Http::fake([
            '*' => Http::response(['status' => 'error'], 500),
        ]);

        $response = $this->client->delete('https://example.com', tries: 3);
        $this->assertEquals(500, $response->status());

        Http::assertSentCount(3);
    }
}
