<?php

namespace Tests\Unit;

use App\Managers\InventoryHealthManager;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLineLayer;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FixSalesOrderLineLayerCacheTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
    }

    /**
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function test_it_can_fix_sales_order_line_layer_cache()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 5);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 5,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertEquals(0, $this->health->salesOrderLinesWithInvalidLayerCacheQuery()->count());

        $this->assertDatabaseHas(SalesOrderLineLayer::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => 5,
        ]);

        SalesOrderLineLayer::each(fn(SalesOrderLineLayer $salesOrderLineLayer) => $salesOrderLineLayer->delete());

        $this->assertEquals(1, $this->health->salesOrderLinesWithInvalidLayerCacheQuery()->count());

        $this->healthManager->fixSalesOrderLineLayerCache($salesOrderLine);

        $this->assertEquals(0, $this->health->salesOrderLinesWithInvalidLayerCacheQuery()->count());
    }
}
