<?php

namespace Tests\Unit;

use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Managers\InventoryHealthManager;
use App\Managers\InventoryMovementManager;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FixOversubscribedFifoLayerTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;
    protected InventoryMovementManager $inventoryMovementManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
        $this->inventoryMovementManager = app(InventoryMovementManager::class);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_identify_and_fix_over_subscribed_fifo_layers()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 11,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertEquals(0, $this->health->overSubscribedFifoLayersQuery()->count());

        BackorderQueue::each(fn(BackorderQueue $backorder) => $backorder->delete());
        $salesOrderLine->getActiveInventoryMovement()->update([
            'quantity' => -11,
        ]);
        $salesOrderLine->getReservedInventoryMovement()->update([
            'quantity' => 11,
        ]);

        $this->assertEquals(1, $this->health->overSubscribedFifoLayersQuery()->count());
        $remainingOversubscribed = $this->healthManager->fixOverSubscribedFifoLayer($fifoLayer);
        $this->assertEquals(0, $remainingOversubscribed);
    }

    /**
     * @throws Throwable
     */
    public function test_it_cant_fix_over_subscribed_fifo_layer_with_fulfillment()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 11,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertEquals(0, $this->health->overSubscribedFifoLayersQuery()->count());

        BackorderQueue::each(fn(BackorderQueue $backorder) => $backorder->delete());
        $salesOrderLine->getActiveInventoryMovement()->update([
            'quantity' => -11,
        ]);
        $salesOrderLine->getReservedInventoryMovement()->update([
            'quantity' => 11,
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 11,
                ]
            ]
        ], false, false);


        $this->assertEquals(1, $this->health->overSubscribedFifoLayersQuery()->count());
        $remainingOversubscribed = $this->healthManager->fixOverSubscribedFifoLayer($fifoLayer);
        $this->assertEquals(1, $remainingOversubscribed);
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     * @throws InsufficientSalesOrderLineQuantityException
     * @throws SalesOrderFulfillmentDispatchException
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function test_it_can_use_stock_take_to_fix_over_subscribed_fifo_layer_with_fulfillment()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);
        $fifoLayer = FifoLayer::first();

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 11,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_OPEN,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => FifoLayer::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => -1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product->id,
            'quantity' => 1,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
        ]);

        $this->assertEquals(0, $this->health->overSubscribedFifoLayersQuery()->count());

        BackorderQueue::each(fn(BackorderQueue $backorder) => $backorder->delete());
        $salesOrderLine->getActiveInventoryMovement()->update([
            'quantity' => -11,
        ]);
        $salesOrderLine->getReservedInventoryMovement()->update([
            'quantity' => 11,
        ]);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 11,
                ]
            ]
        ], false, false);


        $this->assertEquals(1, $this->health->overSubscribedFifoLayersQuery()->count());
        $remainingOversubscribed = $this->healthManager->fixOverSubscribedFifoLayer($fifoLayer);
        $this->assertEquals(1, $remainingOversubscribed);

        $this->healthManager->createStockTakesForOverages(collect([$fifoLayer]), 'Test notes');

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'notes' => 'Test notes',
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product->id,
            'qty_counted' => 0,
            'snapshot_inventory' => -$remainingOversubscribed,
        ]);

        $remainingOversubscribed = $this->healthManager->fixOverSubscribedFifoLayer($fifoLayer);
        $this->assertEquals(0, $remainingOversubscribed);
    }
}
