<?php

namespace Tests\Unit;

use App\Actions\InventoryManagement\CreateSalesOrderReservation;
use App\Managers\InventoryHealthManager;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLineLayer;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FixNonInventoryProductsWithInventoryMovementsTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;
    protected CreateSalesOrderReservation $createSalesOrderReservation;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
        $this->createSalesOrderReservation = app(CreateSalesOrderReservation::class);
    }


    /**
     * @throws Throwable
     */
    public function test_it_can_fix_inventory_movements_for_non_inventory_products()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        $product->setInitialInventory($warehouse->id, 10);

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 15,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        ($this->createSalesOrderReservation)($salesOrder->salesOrderLines->first());

        $product->type = Product::TYPE_BUNDLE;
        $product->save();

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'fulfilled_quantity' => 10,
        ]);
        $this->assertDatabaseHas(BackorderQueue::class, [
            'sales_order_line_id' => $salesOrderLine->id,
            'backordered_quantity' => 5,
        ]);
        $this->assertEquals(5, InventoryMovement::count());
        $this->assertEquals(1, $this->health->getNonInventoryProductsWithInventoryMovementsQuery()->count());

        $this->healthManager->fixNonInventoryProductWithInventoryMovements($product);

        $this->assertEquals(0, $this->health->getNonInventoryProductsWithInventoryMovementsQuery()->count());

        $this->assertDatabaseEmpty(FifoLayer::class);
        $this->assertDatabaseEmpty(BackorderQueue::class);
        $this->assertDatabaseEmpty(InventoryMovement::class);
        $this->assertDatabaseEmpty(SalesOrderLineLayer::class);
    }
}
