<?php

namespace Tests\Unit;

use App\Managers\InventoryHealthManager;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Repositories\InventoryHealthRepository;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FixMissingSalesOrderMovementsTest extends TestCase
{
    use FastRefreshDatabase;

    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;

    protected function setUp(): void
    {
        parent::setUp();

        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
    }

    public static function inStockProvider(): array
    {
        return [
            [true],
            [false],
        ];
    }

    /**
     * @dataProvider inStockProvider
     * @throws Throwable
     */
    public function test_it_can_identify_and_fix_sales_order_lines_without_movements(bool $inStock)
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();

        if ($inStock) {
            $product->setInitialInventory($warehouse->id, 10);
        }

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'quantity' => 1,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $this->assertDatabaseMissing(InventoryMovement::class, [
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrderLine->id,
        ]);

        $salesOrderLinesMissingMovements = $this->health->salesOrderLinesMissingMovementsQuery(InventoryMovement::INVENTORY_STATUS_ACTIVE)->get();
        $this->assertCount(1, $salesOrderLinesMissingMovements);

        $this->healthManager->createSalesOrderReservation($salesOrderLine);
        $salesOrderLinesMissingMovements = $this->health->salesOrderLinesMissingMovementsQuery(InventoryMovement::INVENTORY_STATUS_ACTIVE)->get();
        $this->assertCount(0, $salesOrderLinesMissingMovements);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrderLine->id,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -1,
            'layer_type' => $inStock ? FifoLayer::class : BackorderQueue::class,
        ]);
    }
}
