<?php

namespace Tests\Unit;

use App\Helpers;
use App\Models\SalesOrder;
use App\Models\User;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;
use Illuminate\Support\Carbon;
use App\Models\Product;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;

class DateTableDateFiltersTest extends TestCase
{
    use FastRefreshDatabase;

    public function test_products_can_be_filtered()
    {
        Sanctum::actingAs(User::factory()->create());

        $yearOldProducts = Product::factory(2)->create([
           'created_at' => Carbon::now()->subYear()->subDay(),
            'name' => 'OldProduct-'.rand(100,999)
        ]);

        $sixMonthProducts = Product::factory(3)->create([
            'created_at' => Carbon::now()->subMonths(rand(1, 6)),
            'name' => 'SixMonthProduct-'.rand(100,999)
        ]);

        $newProducts = Product::factory(10)->create([
            'created_at' => Carbon::now(),
            'name' => 'NewProduct-'.rand(100,999)
        ]);

        // Assert Filtering older data
        $dateLocal = Helpers::dateUtcToLocal(Carbon::now()->subMonth()->addDay());

        $requestParams = [
            'limit' => 20,
            'filters' => json_encode([
                'conjunction' => 'and',
                'filterSet' => [[
                    'id' => 28875211,
                    'column' => 'created_at',
                    'operator' => '<',
                    'dateOption' => 'exactDate',
                    'headerValue' => 'created_at',
                    'value' => $dateLocal
                ]]
            ]),
            'sortObjs' => json_encode([]),
            'page' => 1,
            'table_specifications' => 0,
            'total' => 0,
            'archived' => 0,
            'visible_only' => 0,
            'included' => json_encode(['sku'])
        ];

        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(5, $response);

        // Assert filtering by new products this week
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 28875211,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'thisWeek',
                'headerValue' => 'created_at',
                'value' => ['mode' => 'thisWeek']
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(10, $response);

        // Assert filtering by new products in past 6 months or so
        $dateLocal = Helpers::dateUtcToLocal(Carbon::now()->subMonths(6)->subDay()->format('Y-m-d'));
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 28875211,
                'column' => 'created_at',
                'operator' => '>',
                'dateOption' => 'exactDate',
                'headerValue' => 'created_at',
                'value' => $dateLocal
            ]]
        ]);

        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(13, $response);

        // Assert filtering by 'created_at' is within 'the past year'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'pastYear',
                'headerValue' => 'created_at',
                'value' => ['mode' => 'pastYear']
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(13, $response);

        // Assert filtering by 'created_at' is within 'the next number of days'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'nextNumberOfDays',
                'headerValue' => 'created_at',
                'value' => [
                    'mode' => 'nextNumberOfDays', // this mode including today
                    'numberOfDays' => 5
                ]
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(10, $response);

        // Assert filtering by 'created_at' is within 'range of tow dates'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'range',
                'headerValue' => 'created_at',
                'value' => [
                    'from' => Carbon::now()->subYear()->subDay()->startOfDay(),
                    'to' => Carbon::now()->addDay()->format('Y-m-d')
                ]
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(15, $response);

        // Assert filtering by 'created_at' is within 'the past number of days'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'pastNumberOfDays',
                'headerValue' => 'created_at',
                'value' => [
                    'mode' => 'pastNumberOfDays',
                    'numberOfDays' => 20
                ]
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(10, $response);

        // Assert filtering by 'created_at' is within 'the past number of minutes'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'pastNumberOfMinutes',
                'headerValue' => 'created_at',
                'value' => [
                    'mode' => 'pastNumberOfMinutes',
                    'numberOfMinutes' => 24 * 60 // 24 hours
                ]
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(10, $response);

        // Assert filtering by 'created_at' is within 'the past number of minutes'
        $requestParams['filters'] = json_encode([
            'conjunction' => 'and',
            'filterSet' => [[
                'id' => 26516007,
                'column' => 'created_at',
                'operator' => 'isWithin',
                'dateOption' => 'pastNumberOfMinutes',
                'headerValue' => 'created_at',
                'value' => [
                    'mode' => 'pastNumberOfMinutes',
                    'numberOfMinutes' => 60
                ]
            ]]
        ]);
        $response = $this->getJson('/api/products?' . http_build_query($requestParams))->decodeResponseJson()['data'];
        $this->assertCount(10, $response);
    }

    public function test_it_can_filter_sales_orders_for_today(): void
    {
        Sanctum::actingAs(User::first());

        SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create([
                'order_date' => Carbon::now(),
            ]);

        SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create([
                'order_date' => Carbon::now()->subDays(5),
            ]);

        $filters = [
            'filterSet' => [
                [
                    'column' => 'order_date',
                    'dateOption' => 'today',
                    'value' => [
                        'mode' => 'today'
                    ]
                ]
            ]
        ];

        $response = $this->getJson(route('sales-orders.index') . '?filters=' . json_encode($filters))
            ->assertOk();

        $this->assertCount(1, $response->json()['data']);
    }

    public function test_it_can_filter_daily_financials_for_today(): void
    {
        Sanctum::actingAs(User::first());

        SalesOrder::factory()
        ->hasSalesOrderLines()
        ->create([
            'order_date' => Carbon::now(),
        ]);

        SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create([
                'order_date' => Carbon::now()->subDays(5),
            ]);

        app(SalesOrderLineFinancialManager::class)->calculate();
        app(DailyFinancialManager::class)->calculate();

        $filters = [
            'filterSet' => [
                [
                    'column' => 'date',
                    'dateOption' => 'today',
                    'value' => [
                        'mode' => 'today'
                    ]
                ]
            ]
        ];

        $response = $this->getJson(route('financials.daily-summary.index') . '?filters=' . json_encode($filters))
            ->assertOk();

        $this->assertCount(1, $response->json()['data']);
    }
}
