<?php
namespace Tests\Unit;

use App\Data\CreateInventoryAdjustmentData;
use App\DTO\PurchaseOrderDto;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Managers\InventoryAdjustmentManager;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\PurchaseOrder\PurchaseOrderManager;
use App\Services\PurchaseOrder\ShipmentManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Artisan;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class CorrectFifoForNegativeReceiptsTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_correct_fifo_for_negative_receipts(): void
    {
        // set up by creating purchase order receipt then a negative adjustment (manually with factory) that is allocated to the wrong fifo layer

        $purchaseOrder = PurchaseOrder::factory()
            ->withLines(quantity: 10)
            ->receiveQuantity(quantity: 10)
            ->create();

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
        $purchaseOrderLineShipmentReceipt = $purchaseOrderLine->purchaseOrderShipmentLines->first()->purchaseOrderShipmentReceiptLines->first();
        $purchaseOrderLineShipmentReceiptFifoLayer = $purchaseOrderLineShipmentReceipt->fifoLayers()->first();
        $product = $purchaseOrderLine->product;

        // Fully use FIFO layer

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, ['product_id' => $product->id, 'quantity' => 10, 'warehouse_id' => $purchaseOrder->destination_warehouse_id])
            ->create();
        $salesOrderLine = $salesOrder->salesOrderLines()->first();
        app(BulkInventoryManager::class)->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfilled_at' => now(),
            'warehouse_id' => $purchaseOrder->destination_warehouse_id,
            'fulfillment_lines' => $salesOrder->salesOrderLines->map(function ($line) {
                return [
                    'quantity' => $line->quantity,
                    'sales_order_line_id' => $line->id,
                ];
            })->toArray(),
        ], false, false);

        // Create alternative fifo layer

        $inventoryAdjustment = app(InventoryAdjustmentManager::class)->createAdjustment(CreateInventoryAdjustmentData::from([
            'product_id' => $product->id,
            'quantity' => 10,
            'unit_cost' => 10,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $purchaseOrder->destination_warehouse_id,
        ]));
        $alternativeFifoLayer = $inventoryAdjustment->fifoLayers->first();

        // Create negative adjustment allocated to the wrong fifo layer

        $negativeInventoryAdjustment = InventoryAdjustment::factory()
            ->create([
                'product_id' => $product->id,
                'quantity' => -3,
                'warehouse_id' => $purchaseOrder->destination_warehouse_id,
                'link_type' => PurchaseOrderLine::class,
                'link_id' => $purchaseOrderLine->id,
            ]);

        $inventoryMovement = InventoryMovement::factory()
            ->create([
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'type' => InventoryMovement::TYPE_ADJUSTMENT,
                'link_type' => InventoryAdjustment::class,
                'link_id' => $negativeInventoryAdjustment->id,
                'layer_type' => FifoLayer::class,
                'layer_id' => $alternativeFifoLayer->id,
                'quantity' => -3,
            ]);

        $alternativeFifoLayer->fulfilled_quantity = 3;
        $alternativeFifoLayer->save();

        $this->assertDatabaseHas(InventoryMovement::class, [
            'link_id' => $negativeInventoryAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'layer_id' => $alternativeFifoLayer->id,
            'quantity' => -3,
        ]);

        Artisan::call('sku:correct-fifo-for-negative-receipts');

        $this->assertDatabaseHas(InventoryMovement::class, [
            'link_id' => $negativeInventoryAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'layer_id' => $purchaseOrderLineShipmentReceiptFifoLayer->id,
            'quantity' => -3,
        ]);
    }
}
