<?php

namespace Tests\Unit;

use App\Actions\ConvertSalesOrderLineToFinancialLine;
use App\Models\FifoLayer;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\NominalCode;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Exception;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ConvertSalesOrderLineToFinancialLineTest extends TestCase
{
    use FastRefreshDatabase;

    protected ConvertSalesOrderLineToFinancialLine $convertSalesOrderLineToFinancialLine;

    protected function setUp(): void
    {
        parent::setUp();
        $this->convertSalesOrderLineToFinancialLine = app(ConvertSalesOrderLineToFinancialLine::class);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_convert_sales_order_line_to_financial_line()
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::first();
        $nominalCode = NominalCode::factory()->create();
        $financialLineType = FinancialLineType::factory()->create();

        $product->setInitialInventory($warehouse->id, 5);

        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines(1, [
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
                'nominal_code_id' => $nominalCode->id,
                'quantity' => 5,
                'amount' => 100,
                'tax_allocation' => 10,
            ])
            ->create([
                'order_status' => SalesOrder::STATUS_CLOSED,
                'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
                'is_tax_included' => true,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(collect([$salesOrderLine]));

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 5,
                ]
            ]
        ], false, false);

        ($this->convertSalesOrderLineToFinancialLine)($salesOrderLine, $financialLineType);

        $this->assertDatabaseEmpty(SalesOrderLine::class);

        $this->assertDatabaseHas(FinancialLine::class, [
            'sales_order_id' => $salesOrder->id,
            'nominal_code_id' => $nominalCode->id,
            'description' => $salesOrderLine->description,
            'quantity' => $salesOrderLine->quantity,
            'amount' => $salesOrderLine->amount,
            'tax_allocation' => $salesOrderLine->tax_allocation,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'fulfilled_quantity' => 0,
        ]);

        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $salesOrder->id,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);
    }
}
