<?php

namespace Tests\Feature;

use App\Jobs\Shopify\ShopifyGetProducts;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\Shopify\ShopifyOrder as ShopifyOrder;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Models\Shopify\ShopifyWebhookEvent;
use App\Repositories\Shopify\ShopifyWebhookEventRepository;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Support\Facades\Queue;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Modules\Shopify\Jobs\ShopifyOrderWebhookEventJob;
use Tests\TestCaseWithoutTransactions;
use Throwable;

class ShopifyWebhookTest extends TestCase
{
    use FastRefreshDatabase;

    private ShopifyProduct $shopifyProduct;

    private Product $product;

    private IntegrationInstance $shopifyIntegrationInstance;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        // Set up shopify integration instance
        $this->shopifyIntegrationInstance = IntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'integration_id' => Integration::where('name', 'Shopify')->first()->id,
                'name' => 'Mamimoo',
            ]);
    }

    public function setUpProduct(): void
    {
        $shopifyProductPayload = [
            'id' => 8286016831797,
            'title' => 'ShipstationTest1',
            'vendor' => 'Mamimoo Baby Clothing',
            'created_at' => '2023-05-10T21:57:18-04:00',
            'handle' => 'shipstationtest1',
            'updated_at' => '2023-06-17T14:11:01-04:00',
            'status' => 'active',
            'tags' => '',
            'variants' => [
                [
                    'id' => 45048073978165,
                    'product_id' => 8286016831797,
                    'title' => 'Default Title',
                    'price' => '5.00',
                    'sku' => 'ShipstationTest1',
                    'created_at' => '2023-05-10T21:57:19-04:00',
                    'updated_at' => '2023-06-17T14:11:01-04:00',
                    'weight' => 0,
                    'weight_unit' => 'lb',
                    'inventory_quantity' => -5,
                    'image_id' => null,
                ],
            ],
            'images' => [],
            'image' => null,
        ];

        $productVariants = ShopifyGetProducts::extractProductVariations($this->shopifyIntegrationInstance, $shopifyProductPayload);

        ShopifyProduct::query()->insert($productVariants[0]);

        /** @var ShopifyProduct $shopifyProduct */
        $shopifyProduct = ShopifyProduct::query()->first();

        $this->shopifyProduct = $shopifyProduct;

        $this->shopifyProduct->createSkuProduct();

        $this->product = $this->shopifyProduct->productListing()->first()->product;
    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_it_can_store_and_process_shopify_order_update_webhook(): void
    {
        $createPayload = [
            'id' => 5379797745973,
            'currency' => 'USD',
            'name' => '#1001',
            'order_number' => '1001',
            'processed_at' => '2023-05-11T10:52:33-04:00',
            'created_at' => '2023-05-11T10:52:34-04:00',
            'updated_at' => '2023-05-11T10:52:35-04:00',
            'shipping_lines' => [],
            'taxes_included' => false,
            'customer' => [
                'name' => 'Rick Jacobs',
                'first_name' => 'Rick',
                'last_name' => 'Jacobs',
                'email' => 'rick@sku.io',
                'default_address' => [
                    'address1' => '500 Westover Drive',
                    'address2' => '#140-2671',
                    'city' => 'Sanford',
                    'province' => 'North Carolina',
                    'province_code' => 'NC',
                    'zip' => '27330',
                    'country_code' => 'US',
                ],
            ],
            'shipping_address' => [
                'address1' => '500 Westover Drive',
                'address2' => '#140-2671',
                'city' => 'Sanford',
                'province' => 'North Carolina',
                'province_code' => 'NC',
                'zip' => '27330',
                'country_code' => 'US',
            ],
            'billing_address' => [
                'address1' => '500 Westover Drive',
                'address2' => '#140-2671',
                'city' => 'Sanford',
                'province' => 'North Carolina',
                'province_code' => 'NC',
                'zip' => '27330',
                'country_code' => 'US',
            ],
            'line_items' => [
                [
                    'id' => 13955632464181,
                    'variant_id' => 45048073978165,
                    'sku' => 'ShipstationTest1',
                    'name' => 'ShipstationTest1',
                    'fulfillable_quantity' => 2,
                    'quantity' => 2,
                    'price' => 5.00,
                    'gift_card' => false,
                    'product_exists' => true,
                    'product_id' => 8286016831797,
                    'tax_lines' => [],
                    'discount_allocations' => [],
                ],
            ],
            'refunds' => [],
        ];

        // Post to webhook endpoint
        $response = $this->postJson(
            route('webhooks.shopify.order_create', [
                $this->shopifyIntegrationInstance->id,
                'token' => config('webhooks.shpfy'),
            ]),
            $createPayload
        );

        (new ShopifyOrderWebhookEventJob($this->shopifyIntegrationInstance, ShopifyWebhook::TOPIC_ORDERS_CREATE, $createPayload))->handle();

        app(ShopifyWebhookEventRepository::class)->process($this->shopifyIntegrationInstance);

        $response->assertOk();

        // Assert that the webhook was stored
        $this->assertDatabaseHas((new ShopifyWebhookEvent())->getTable(), [
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'topic' => 'orders',
            'unique_id' => 5379797745973,
            'json_data' => json_encode($createPayload),
        ]);

        app(ShopifyWebhookEventRepository::class)->process($this->shopifyIntegrationInstance);

        $this->assertEquals(
            0,
            ShopifyWebhookEvent::query()
                ->whereNull('processed_at')
                ->count()
        );

        $this->assertDatabaseHas((new ShopifyOrder())->getTable(), [
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'name' => '#1001',
            'json_object' => json_encode($createPayload),
        ]);
    }

    public function test_it_can_process_shopify_webhooks_in_correct_order(): void
    {
        $uniqueId = 123;

        $payloadDate = now()->addDay();

        $repo = app(ShopifyWebhookEventRepository::class);

        $webhookEventNewestPayload = ShopifyWebhookEvent::factory()->create([
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'topic' => ShopifyWebhook::TOPIC_ORDERS_UPDATED,
            'unique_id' => $uniqueId,
            'payload_date' => $payloadDate,
            'processed_at' => null,
        ]);

        // Oldest Payload should not be processed even as long as a newer payload exists
        $webhookEventOldestPayload = ShopifyWebhookEvent::factory()->create([
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'topic' => ShopifyWebhook::TOPIC_ORDERS_UPDATED,
            'unique_id' => $uniqueId,
            'payload_date' => $payloadDate->copy()->subDay(),
            'processed_at' => null,
        ]);

        $needsProcessing = $repo->getWebhookEventsNeedingProcessing($this->shopifyIntegrationInstance);

        $this->assertCount(1, $needsProcessing);
        $this->assertEquals($webhookEventNewestPayload->id, $needsProcessing[0]->id);

        $webhookEventNewestPayload->update(['processed_at' => now()]);

        $needsProcessing = $repo->getWebhookEventsNeedingProcessing($this->shopifyIntegrationInstance);

        $this->assertCount(0, $needsProcessing);

        $webhookEventNewestPayload->delete();

        $needsProcessing = $repo->getWebhookEventsNeedingProcessing($this->shopifyIntegrationInstance);

        $this->assertCount(1, $needsProcessing);
        $this->assertEquals($webhookEventOldestPayload->id, $needsProcessing[0]->id);
    }
}
