<?php

namespace Tests\Feature;

use App\Jobs\Shopify\ShopifyGetProducts;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder as ShopifyOrder;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Models\Shopify\ShopifyWebhookEvent;
use App\Models\Tag;
use App\Models\Warehouse;
use App\Repositories\Shopify\ShopifyWebhookEventRepository;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Queue;
use Modules\Shopify\Jobs\ShopifyOrderWebhookEventJob;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ShopifyOrderUpdateTest extends TestCase
{
    use FastRefreshDatabase;

    private Warehouse $warehouse;

    private ShopifyProduct $shopifyProduct;

    private ShopifyOrder $shopifyOrder;

    private SalesOrder $salesOrder;

    private Product $product;

    private array $createPayload;

    private IntegrationInstance $shopifyIntegrationInstance;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        // Set up shopify integration instance
        $this->shopifyIntegrationInstance = IntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create([
                'integration_id' => Integration::where('name', 'Shopify')->first()->id,
                'name' => 'Mamimoo',
            ]);

        $this->warehouse = Warehouse::first();
    }

    public function setUpProduct(): void
    {
        $shopifyProductPayload = [
            'id' => 8286016831797,
            'title' => 'ShipstationTest1',
            'vendor' => 'Mamimoo Baby Clothing',
            'created_at' => '2023-05-10T21:57:18-04:00',
            'handle' => 'shipstationtest1',
            'updated_at' => '2023-06-17T14:11:01-04:00',
            'status' => 'active',
            'tags' => '',
            'variants' => [
                [
                    'id' => 45048073978165,
                    'product_id' => 8286016831797,
                    'title' => 'Default Title',
                    'price' => '5.00',
                    'sku' => 'ShipstationTest1',
                    'created_at' => '2023-05-10T21:57:19-04:00',
                    'updated_at' => '2023-06-17T14:11:01-04:00',
                    'weight' => 0,
                    'weight_unit' => 'lb',
                    'inventory_quantity' => -5,
                    'image_id' => null,
                    'requires_shipping' => true,
                ],
            ],
            'images' => [],
            'image' => null,
        ];

        $productVariants = ShopifyGetProducts::extractProductVariations($this->shopifyIntegrationInstance, $shopifyProductPayload);

        ShopifyProduct::query()->insert($productVariants[0]);

        /** @var ShopifyProduct $shopifyProduct */
        $shopifyProduct = ShopifyProduct::query()->first();

        $this->shopifyProduct = $shopifyProduct;

        $this->shopifyProduct->createSkuProduct();

        $this->product = $this->shopifyProduct->productListing()->first()->product;
    }

    public function setUpOrder(): void
    {
        $this->createPayload = [
            'id' => 5379797745973,
            'currency' => 'USD',
            'name' => '#1001',
            'order_number' => '1001',
            'processed_at' => '2023-05-11T10:52:33-04:00',
            'created_at' => '2023-05-11T10:52:34-04:00',
            'updated_at' => '2023-05-11T10:52:35-04:00',
            'shipping_lines' => [],
            'taxes_included' => false,
            'customer' => [
                'name' => 'Rick Jacobs',
                'first_name' => 'Rick',
                'last_name' => 'Jacobs',
                'email' => 'rick@sku.io',
                'default_address' => [
                    'address1' => '500 Westover Drive',
                    'address2' => '#140-2671',
                    'city' => 'Sanford',
                    'province' => 'North Carolina',
                    'province_code' => 'NC',
                    'zip' => '27330',
                    'country_code' => 'US',
                ],
            ],
            'shipping_address' => [
                'first_name' => 'Rick',
                'last_name' => 'Jacobs',
                'address1' => '500 Westover Drive',
                'address2' => '#140-2671',
                'city' => 'Sanford',
                'province' => 'North Carolina',
                'province_code' => 'NC',
                'zip' => '27330',
                'country_code' => 'US',
            ],
            'billing_address' => [
                'first_name' => 'Rick',
                'last_name' => 'Jacobs',
                'address1' => '500 Westover Drive',
                'address2' => '#140-2671',
                'city' => 'Sanford',
                'province' => 'North Carolina',
                'province_code' => 'NC',
                'zip' => '27330',
                'country_code' => 'US',
            ],
            'line_items' => [
                [
                    'id' => 13955632464181,
                    'variant_id' => 45048073978165,
                    'sku' => 'ShipstationTest1',
                    'name' => 'ShipstationTest1',
                    'fulfillable_quantity' => 2,
                    'quantity' => 2,
                    'price' => 5.00,
                    'gift_card' => false,
                    'product_exists' => true,
                    'product_id' => 8286016831797,
                    'tax_lines' => [],
                    'discount_allocations' => [],
                ],
            ],
            'refunds' => [],
        ];

        // Post to webhook endpoint
        $response = $this->postJson(
            route('webhooks.shopify.order_create', [
                $this->shopifyIntegrationInstance->id,
                'token' => config('webhooks.shpfy'),
            ]),
            $this->createPayload
        );

        (new ShopifyOrderWebhookEventJob($this->shopifyIntegrationInstance, ShopifyWebhook::TOPIC_ORDERS_UPDATED, $this->createPayload))->handle();

        $response->assertOk();

        // Assert that the webhook was stored
        $this->assertDatabaseHas((new ShopifyWebhookEvent())->getTable(), [
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'topic' => 'orders',
            'unique_id' => 5379797745973,
            'json_data' => json_encode($this->createPayload),
        ]);

        app(ShopifyWebhookEventRepository::class)->process($this->shopifyIntegrationInstance);

        $this->assertEquals(
            0,
            ShopifyWebhookEvent::query()
                ->whereNull('processed_at')
                ->count()
        );

        $payload = $this->createPayload;

        $this->assertDatabaseHas((new ShopifyOrder())->getTable(), [
            'integration_instance_id' => $this->shopifyIntegrationInstance->id,
            'name' => '#1001',
            'updatedAtUtc' => Carbon::parse($payload['updated_at'])->setTimezone('UTC')->toDateTimeString(),
            'createdAtUtc' => Carbon::parse($payload['created_at'])->setTimezone('UTC')->toDateTimeString(),
            'processedAtUtc' => Carbon::parse($payload['processed_at'])->setTimezone('UTC')->toDateTimeString(),
            'json_object' => json_encode($this->createPayload),
        ]);

        /** @var ShopifyOrder $shopifyOrder */
        $shopifyOrder = ShopifyOrder::query()->first();

        // Create sku order
        $this->shopifyOrder = $shopifyOrder;

        $this->shopifyOrder->createSKUOrder();

        /** @var SalesOrder $salesOrder */
        $salesOrder = $this->shopifyOrder->salesOrder()->first();

        $this->salesOrder = $salesOrder;

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 2,
            'canceled_quantity' => 0,
            'amount' => 5.00,
        ]);
    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_it_can_update_sales_order_line_when_shopify_quantity_partially_canceled_before_mapping(): void
    {
        $this->setUpOrder();
        $updatePayload = $this->createPayload;
        $updatePayload['line_items'][0]['fulfillable_quantity'] = 1;
        $updatePayload['refunds'] = [
            [
                'id' => 949368258869,
                'created_at' => '2023-06-17T13:57:58-04:00',
                'order_id' => 5379797745973,
                'processed_at' => '2023-06-17T13:57:58-04:00',
                'restock' => true,
                'order_adjustments' => [],
                'transactions' => [],
                'refund_line_items' => [
                    [
                        'id' => 563200524597,
                        'line_item_id' => 13955632464181,
                        'quantity' => 1,
                        'restock_type' => 'cancel',
                        'subtotal' => 5,
                        'line_item' => [
                            'id' => 13955632464181,
                        ],
                    ],
                ],
            ],
        ];
        $updatePayload['updated_at'] = '2023-05-11T10:52:36-04:00';
        $updatePayload['tags'] = 'tag1, tag2';
        $this->postJson(
            route('webhooks.shopify.order_updated', [
                $this->shopifyIntegrationInstance->id,
                'token' => config('webhooks.shpfy'),
            ]),
            $updatePayload
        )->assertOk();

        (new ShopifyOrderWebhookEventJob($this->shopifyIntegrationInstance, ShopifyWebhook::TOPIC_ORDERS_UPDATED, $updatePayload))->handle();

        app(ShopifyWebhookEventRepository::class)->process($this->shopifyIntegrationInstance);

        // Needed to update the order
        $this->shopifyOrder->refresh();

        $salesOrder = $this->shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 1,
            'canceled_quantity' => 1,
            'amount' => 5.00,
        ]);

        $this->assertEquals(
            'tag1',
            $salesOrder->tags()->first()->name
        );

        $this->assertEquals(
            'tag2',
            $salesOrder->tags()->offset(1)->first()->name
        );
    }

    /**
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function test_it_can_process_multiple_updates_to_sales_order_line_when_shopify_order_line_partially_cancelled(): void
    {
        $this->setUpProduct();
        $this->setUpOrder();
        $updatePayload = $this->createPayload;
        $updatePayload['line_items'][0]['fulfillable_quantity'] = 1;
        $updatePayload['refunds'] = [
            [
                'id' => 949368258869,
                'created_at' => '2023-06-17T13:57:58-04:00',
                'order_id' => 5379797745973,
                'processed_at' => '2023-06-17T13:57:58-04:00',
                'restock' => true,
                'order_adjustments' => [],
                'transactions' => [],
                'refund_line_items' => [
                    [
                        'id' => 563200524597,
                        'line_item_id' => 13955632464181,
                        'quantity' => 1,
                        'restock_type' => 'cancel',
                        'subtotal' => 5,
                        'line_item' => [
                            'id' => 13955632464181,
                        ],
                    ],
                ],
            ],
        ];
        $updatePayload['updated_at'] = '2023-05-11T10:52:36-04:00';
        $this->postJson(
            route('webhooks.shopify.order_updated', [
                $this->shopifyIntegrationInstance->id,
                'token' => config('webhooks.shpfy'),
            ]),
            $updatePayload
        )->assertOk();

        (new ShopifyOrderWebhookEventJob($this->shopifyIntegrationInstance, ShopifyWebhook::TOPIC_ORDERS_UPDATED, $updatePayload))->handle();

        app(ShopifyWebhookEventRepository::class)->process($this->shopifyIntegrationInstance);

        // Needed to update the order
        $this->shopifyOrder->refresh();

        $this->shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 1,
            'canceled_quantity' => 1,
            'amount' => 5.00,
        ]);

        $this->shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 1,
            'canceled_quantity' => 1,
            'amount' => 5.00,
        ]);

    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_it_can_update_sales_order_line_when_shopify_quantity_partially_canceled_after_mapping_and_fulfillment(): void
    {
        /*
         * https://siberventures.atlassian.net/wiki/spaces/ORD3/pages/1724350465/Shopify+Order+Updates+when+Fulfilled
         * Still deciding on desired logic
         */
        //$this->markTestSkipped('Logic TBD');
        $this->setUpProduct();

        // Get stock in
        $this->product->setInitialInventory($this->warehouse->id, 10, 5.00);

        $this->setUpOrder();

        // Fulfill order submitted

        // This will create a sales order fulfillment in "submitted" status, which is eligible for deletion upon cancellation of the line items
        $fulfillmentManager = app(FulfillmentManager::class);
        $fulfillmentManager->fulfill($this->salesOrder, [
            'warehouse_id' => $this->warehouse->id,
            'fulfillment_lines' => [
                [
                    'quantity' => 2,
                    'sales_order_line_id' => $this->salesOrder->salesOrderLines()->first()->id,
                ],
            ],
        ], false, false);

        $this->assertDatabaseHas((new SalesOrderFulfillment())->getTable(), [
            'sales_order_id' => $this->salesOrder->id,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
        ]);

        $this->refundShopifyOrder();

        $this->shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 0,
            'canceled_quantity' => 2,
            'amount' => 5.00,
        ]);

        // Fulfillment status "submitted" gets deleted on cancellation
        $this->assertDatabaseEmpty((new SalesOrderFulfillment())->getTable());

        /*
         * We run this again to make sure subsequent updates don't cause issues
         */
        $this->shopifyOrder->updateSKUOrder();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'description' => 'ShipstationTest1',
            'quantity' => 0,
            'canceled_quantity' => 2,
            'amount' => 5.00,
        ]);

        // When an order is cancelled, it should be closed and fulfilled

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $this->salesOrder->id,
            'order_status' => SalesOrder::STATUS_CLOSED,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);

        // Shopify update to no refunds, which should uncancel

        /*
         * TODO: Uncancel functionality currently disabled until logic is updated in Shopify/Order::handleCancellations and HandleSalesOrderLineCancellations
         *  The problem is if quantity_to_cancel is 0, currently it would mark the order as out of sync, but right now the quantity_to_cancel from handleCancellations will
         *  always be 0 if nothing is happening.  We need to address the case of a true uncancel situation and not set quantity_to_cancel to 0 in that case.  Or
         *  address the issue in HandleSalesOrderLineCancellations by maybe comparing the quantity to cancel to the existing canceled quantity.  Long term I think
         *  we should not modify quantity and instead have a stored generated field called quantity_net_cancellations or something like that.
         */
        //        $updatePayload = $this->createPayload;
        //        $updatePayload['line_items'][0]['fulfillable_quantity'] = 1;
        //        $updatePayload['refunds'] = [];
        //
        //        ShopifyDownloadOrderbulkSaveShopifyOrderToSkuDB(
        //            $this->shopifyIntegrationInstance->id,
        //            [$updatePayload],
        //            'user',
        //        );
        //
        //        // Important to refresh order here
        //        $this->shopifyOrder->refresh();
        //
        //        $this->shopifyOrder->updateSKUOrder();
        //        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
        //            'description' => 'ShipstationTest1',
        //            'quantity' => 2,
        //            'canceled_quantity' => 0,
        //            'amount' => 5.00,
        //        ]);
        //
        //        // Now we want to do a new fulfillment but make it fulfilled instead of submitted
        //
        //        $fulfillmentManager = app(FulfillmentManager::class);
        //        $fulfillmentManager->fulfill($this->salesOrder, [
        //            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
        //            'warehouse_id' => $this->warehouse->id,
        //            'fulfillment_lines' => [
        //                [
        //                    'quantity' => 2,
        //                    'sales_order_line_id' => $this->salesOrder->salesOrderLines()->first()->id,
        //                ],
        //            ],
        //        ], false, false);
        //
        //        $this->assertDatabaseHas((new SalesOrderFulfillment())->getTable(), [
        //            'sales_order_id' => $this->salesOrder->id,
        //            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
        //        ]);
        //
        //        // Now when we try to cancel the shopify order, the cancellation won't go through and the appropriate tag added
        //
        //        $this->refundShopifyOrder();
        //
        //        $this->shopifyOrder->updateSKUOrder();
        //
        //        // Not cancelled due to fulfillment
        //        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
        //            'description' => 'ShipstationTest1',
        //            'quantity' => 2,
        //            'canceled_quantity' => 0,
        //            'amount' => 5.00,
        //        ]);
        //
        //        $this->assertEquals(1, Tag::containing('fulfilled-on-cancelled-line')->count());
    }

    private function refundShopifyOrder(): void
    {
        $updatePayload = $this->createPayload;
        $updatePayload['line_items'][0]['fulfillable_quantity'] = 1;
        $updatePayload['refunds'] = [
            [
                'id' => 949368258869,
                'created_at' => '2023-06-17T13:57:58-04:00',
                'order_id' => 5379797745973,
                'processed_at' => '2023-06-17T13:57:58-04:00',
                'restock' => true,
                'order_adjustments' => [],
                'transactions' => [],
                'refund_line_items' => [
                    [
                        'id' => 563200524597,
                        'line_item_id' => 13955632464181,
                        'quantity' => 2,
                        'restock_type' => 'cancel',
                        'subtotal' => 5,
                        'line_item' => [
                            'id' => 13955632464181,
                        ],
                    ],
                ],
            ],
        ];

        ShopifyDownloadOrder::bulkSaveShopifyOrderToSkuDB(
            $this->shopifyIntegrationInstance->id,
            [$updatePayload],
            'user',
        );

        // Needed to update the order
        $this->shopifyOrder->refresh();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_process_returns(): void
    {
        /*
         * 1. Create shopify order
         * 2. Create sku order from shopify order
         * 3. Update shopify order payload
         * 4. Update sku order from shopify order
         *
         */

        $this->setUpProduct();
        $this->product->setInitialInventory($this->warehouse->id, 2, 5.00);
        $this->setUpOrder();

        app(FulfillmentManager::class)->fulfill($this->salesOrder, [
            'warehouse_id' => $this->warehouse->id,
            'fulfillment_lines' => [
                [
                    'quantity' => 2,
                    'sales_order_line_id' => $this->salesOrder->salesOrderLines->first()->id,
                ],
            ],
        ], false, false);

        $updatePayload = $this->shopifyOrder->json_object;

        $updatePayload['fulfillments'] = [
            [
                'id' => 255858046109,
                'status' => 'success',
                'created_at' => Carbon::now()->toDateTimeString(),
                'line_items' => [
                    [
                        'id' => 13955632464181,
                        'quantity' => 2,
                        'fulfillable_quantity' => 0,
                        'fulfillment_status' => 'fulfilled',
                        'line_item' => [
                            'id' => 13955632464181,
                        ],
                    ],
                ],
            ],
        ];

        $updatePayload['refunds'] = [
            [
                'id' => 949368258869,
                'created_at' => '2023-06-17T13:57:58-04:00',
                'order_id' => 5379797745973,
                'processed_at' => '2023-06-17T13:57:58-04:00',
                'restock' => true,
                'order_adjustments' => [],
                'transactions' => [],
                'refund_line_items' => [
                    [
                        'id' => 563200524597,
                        'line_item_id' => 13955632464181,
                        'quantity' => 2,
                        'restock_type' => 'return',
                        'line_item' => [
                            'id' => 13955632464181,
                        ],
                        'subtotal' => 5.00,
                    ],
                ],
            ],
        ];

        $this->shopifyOrder->json_object = $updatePayload;
        $this->shopifyOrder->save();
        $this->shopifyOrder->refresh();

        $this->shopifyOrder->handleRefunds();

        $this->assertDatabaseHas((new SalesCreditLine())->getTable(), [
            'product_id' => $this->product->id,
        ]);
    }
}
