<?php

namespace Tests\Feature\SalesOrders;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\DeleteProductInventoryJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Managers\ProductInventoryManager;
use App\Models\Customer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Services\SalesOrder\Fulfillments\Dispatchers\ShipstationDispatcher;
use App\Services\SalesOrder\Fulfillments\Dispatchers\StarshipitDispatcher;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\TestResponse;
use Laravel\Sanctum\Sanctum;
use Mockery\MockInterface;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class FulfillSalesOrderTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_update_a_sales_order_fulfillment(): void
    {
        Queue::fake();

        // Create the sales order.
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();
        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 1,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        /** @var SalesOrderRepository $orders */
        $orders = app(SalesOrderRepository::class);
        $fulfillment = $orders->findFulfillmentById($data['id'], ['salesOrderFulfillmentLines']);

        // Update the fulfillment
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'id' => $fulfillment->salesOrderFulfillmentLines->first()->id,
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->putJson('/api/sales-order-fulfillments/'.$data['id'], $payload);
        $response->assertSuccessful();

        $this->assertDatabaseHas('sales_orders', [
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);

        // Fulfillment and line should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'sales_order_id' => $order->id,
        ]);
        $this->assertDatabaseCount('sales_order_fulfillment_lines', 1);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $salesOrderLine->id,
        ]);

        // Inventory movements should be created
        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'link_id' => $salesOrderLine->id,
            'link_type' => $salesOrderLine::class,
            'quantity' => -$salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'link_id' => $salesOrderLine->id,
            'link_type' => $salesOrderLine::class,
            'quantity' => $salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'link_type' => SalesOrderFulfillmentLine::class,
            'quantity' => -$salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
    }

    public function test_it_can_submit_fulfillment_to_shipstation(): void
    {

        // Mock Starshipit service
        $this->mock(ShipstationDispatcher::class, function (MockInterface $mock) {
            $mock->shouldReceive('dispatchFulfillmentToProvider')->once()->andReturn();
        });

        // Create the sales order.
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPSTATION,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();
    }

    public function test_it_can_submit_fulfillment_to_starshipit(): void
    {

        // Mock Starshipit service
        $this->mock(StarshipitDispatcher::class, function (MockInterface $mock) {
            $mock->shouldReceive('dispatchFulfillmentToProvider')->once()->andReturn();
        });

        // Create the sales order.
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_STARSHIPIT,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();
    }

    public function test_it_can_recalculate_product_inventory_cache_after_fulfillment(): void
    {
        Queue::fake();

        // Create the sales order.
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();

        (new ProductInventoryManager(null, true, false, true))->updateProductInventoryAndAvgCost();

        $this->assertDatabaseCount('products_inventory', 2);
        $this->assertDatabaseHas('products_inventory', [
            'warehouse_id' => $salesOrderLine->warehouse_id,
            'inventory_reserved' => 0,
            'inventory_total' => 0,
        ]);
    }

    public function test_it_can_fulfill_dropship_sales_orders(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create([
            'auto_fulfill_dropship' => true,
        ]);

        Warehouse::factory()->create([
            'dropship_enabled' => true,
            'supplier_id' => $supplier->id,
        ]);

        /** @var Product $product */
        $product = Product::factory()->create([
            'is_dropshippable' => true,
        ]);

        SupplierProduct::factory()->create([
            'product_id' => $product->id,
            'supplier_id' => $supplier->id,
        ]);

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::DROPSHIP->value,
                ],
            ],
        ];
        // Create dropship order.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Fulfill dropship order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
        ];
        $response = $this->postJson('/api/sales-orders/'.$data['id'].'/fulfill', $payload);
        $response->assertSuccessful();

        // Sales order should be closed
        $this->assertDatabaseHas('sales_orders', [
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);

        // Fulfillment should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseCount('sales_order_fulfillment_lines', 1);

        // No Fulfillment movement should be created.
        $this->assertDatabaseCount('inventory_movements', 0);

        // Dropship PO should be closed
        $this->assertDatabaseHas('purchase_orders', [
            'order_status' => PurchaseOrder::STATUS_CLOSED,
            'receipt_status' => PurchaseOrder::RECEIPT_STATUS_RECEIVED,
        ]);

        // Purchase shipments and receipts should be created.
        $this->assertDatabaseCount('purchase_order_shipments', 1);
        $this->assertDatabaseCount('purchase_order_shipment_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_lines', [
            'quantity' => 2,
        ]);

        $this->assertDatabaseCount('purchase_order_shipment_receipts', 1);
        $this->assertDatabaseCount('purchase_order_shipment_receipt_lines', 1);
        $this->assertDatabaseHas('purchase_order_shipment_receipt_lines', [
            'quantity' => 2,
        ]);
    }

    public function test_it_can_fulfill_a_sales_order(): void
    {
        Queue::fake();

        // Create the sales order.
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();
        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        $this->assertDatabaseHas('sales_orders', [
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);

        // Fulfillment and line should be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'sales_order_id' => $order->id,
        ]);
        $this->assertDatabaseCount('sales_order_fulfillment_lines', 1);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $salesOrderLine->id,
        ]);

        // Inventory movements should be created
        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'link_id' => $salesOrderLine->id,
            'link_type' => $salesOrderLine::class,
            'quantity' => -$salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'link_id' => $salesOrderLine->id,
            'link_type' => $salesOrderLine::class,
            'quantity' => $salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'link_type' => SalesOrderFulfillmentLine::class,
            'quantity' => -$salesOrderLine->quantity,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
    }

    public function test_it_can_cancel_fulfillment_if_shipping_address_changed(): void
    {
        Queue::fake();

        // Create the Shipstation integration instance
        IntegrationInstance::factory()->create([
            'integration_id' => Integration::where('name', Integration::NAME_SHIPSTATION)->firstOrFail()->id,
            'connection_settings' => [
                'apiKey' => ' ',
                'apiSecret' => ' ',
            ],
        ]);

        // Mock the Starshipit service
        $this->mock(ShipstationDispatcher::class, function (MockInterface $mock) {
            $mock->shouldReceive('dispatchFulfillmentToProvider')->once()->andReturn();
        });

        // Create the sales order
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->reserved();
        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $order->salesOrderLines->first();

        // Fulfill the sales order
        $payload = [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPSTATION,
            'fulfilled_at' => now(),
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                ],
            ],
        ];

        $response = $this->postJson('/api/sales-orders/'.$order->id.'/fulfill', $payload);
        $response->assertSuccessful();

        $data = json_decode($response->getContent(), true)['data'];

        // Assert that the fulfillment and line are created
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'sales_order_id' => $order->id,
        ]);
        $this->assertDatabaseCount('sales_order_fulfillment_lines', 1);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $salesOrderLine->id,
        ]);

        // Update the shipping address without any changes
        $payload = [
            'shipping_address' => [
                'name' => $order->shippingAddress->name,
                'company' => $order->shippingAddress->company,
                'email' => $order->shippingAddress->email,
                'phone' => $order->shippingAddress->phone,
                'address1' => $order->shippingAddress->address1,
                'address2' => $order->shippingAddress->address2,
                'address3' => $order->shippingAddress->address3,
                'city' => $order->shippingAddress->city,
                'province' => $order->shippingAddress->province,
                'province_code' => $order->shippingAddress->province_code,
                'zip' => $order->shippingAddress->zip,
                'country_code' => $order->shippingAddress->country_code,
                'country' => $order->shippingAddress->country,
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$order->id, $payload);
        $response->assertSuccessful();

        // Assert that fulfillments are not cancelled when the address is not changed
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'sales_order_id' => $order->id,
        ]);

        // Change the shipping address
        $payload = [
            'shipping_address' => [
                'address1' => '123 Old Street',
                'city' => 'Old City',
                'province' => 'Old Province',
                'zip' => '12345',
            ],
        ];

        $response = $this->putJson('/api/sales-orders/'.$order->id, $payload);
        $response->assertSuccessful();

        // Assert that fulfillments are cancelled when the address is changed
        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseMissing('sales_order_fulfillments', [
            'sales_order_id' => $order->id,
        ]);
    }

    public function test_that_over_fulfillments_create_negative_reservations(): void{

        /** @var Product $product */
        $product = Product::factory()->create();
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        // Create positive adjustment with stock
        $this->createAdjustment($product, $warehouse, 10);

        // Create sales order
        $response = $this->createOpenSalesOrder($product, $warehouse, 5);

        // Create over-fulfillment
        $this->postJson(route('sales-orders.fulfill', $response->json('data.id')), [
            'fulfilled_at' => now(),
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 7,
                ]
            ]
        ])->assertSuccessful();

        // Order status should be fulfilled, with notes and the original reservations increased.
        $this->assertDatabaseHas('sales_orders', [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
        ]);

        // Inventory movements should stay the same
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'quantity' => -5,
            'warehouse_id' => $warehouse->id,
            'link_type' => SalesOrderLine::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $response->json('data.item_info.0.sales_order_line_id'),
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'quantity' => 5,
            'warehouse_id' => $warehouse->id,
            'link_type' => SalesOrderLine::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_id' => $response->json('data.item_info.0.sales_order_line_id'),
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'quantity' => -5,
            'warehouse_id' => $warehouse->id,
            'link_type' => SalesOrderFulfillmentLine::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED
        ]);

        // There should be a note on the sales order
        $this->assertDatabaseCount('notes', 1);
        $this->assertDatabaseHas('notes', [
            'link_id' => $response->json('data.id'),
            'link_type' => SalesOrder::class,
        ]);

        // The sales order fulfillment line should be created
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'quantity' => 5,
        ]);

        // Negative inventory adjustment should be created.
        $this->assertDatabaseCount('inventory_adjustments', 2);
        $this->assertDatabaseHas('inventory_adjustments', [
            'product_id' => $product->id,
            'quantity' => -2,
            'warehouse_id' => $warehouse->id
        ]);

    }


    public function test_over_fulfillment_will_fail_when_stock_does_not_exist_to_create_negative_adjustment(): void{

        /** @var Product $product */
        $product = Product::factory()->create();
        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        // Create positive adjustment with stock
        $this->createAdjustment($product, $warehouse, 5);

        // Create sales order
        $response = $this->createOpenSalesOrder($product, $warehouse, 5);

        // Create over-fulfillment
        $this->postJson(route('sales-orders.fulfill', $response->json('data.id')), [
            'fulfilled_at' => now(),
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 7,
                ]
            ]
        ])->assertBadRequest();

        // Order status should be fulfilled, with notes and the original reservations increased.
        $this->assertDatabaseHas('sales_orders', [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
        ]);

        // Inventory movements should stay the same
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'quantity' => -5,
            'warehouse_id' => $warehouse->id,
            'link_type' => SalesOrderLine::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $response->json('data.item_info.0.sales_order_line_id'),
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'quantity' => 5,
            'warehouse_id' => $warehouse->id,
            'link_type' => SalesOrderLine::class,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'link_id' => $response->json('data.item_info.0.sales_order_line_id'),
        ]);


        // Fulfillment should not be created.
        $this->assertDatabaseCount('sales_order_fulfillments', 0);

    }


    private function createOpenSalesOrder(Product $product, Warehouse $warehouse, int $quantity): TestResponse
    {
        return $this->postJson(route('sales-orders.store'), [
            'order_date' => now(),
            'sales_order_number' => 'SO-TEST',
            'currency_code' => 'USD',
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'amount' => 10,
                    'description' => 'Test',
                    'warehouse_id' => $warehouse->id
                ]
            ]
        ])->assertSuccessful();
    }

    private function createAdjustment(Product $product, Warehouse $warehouse, int $quantity): void
    {
        $this->postJson(route('inventory-adjustments.store'), [
            'product_id' => $product->id,
            'quantity' => $quantity,
            'unit_cost' => 5,
            'warehouse_id' => $warehouse->id,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE
        ])->assertSuccessful();
    }
}
