<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\Shopify\ShopifyCreateSkuProducts;
use App\Models\Address;
use App\Models\Customer;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\Setting;
use App\Models\ShippingMethod;
use App\Models\Shopify\ShopifyOrder as ShopifyOrder;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseOld;
use Throwable;

class SalesOrderTest extends TestCase
{
    use FastRefreshDatabase;

    protected static $user;

    protected static $customer;

    protected static $shippingMethod;

    public function test_tags(): void
    {
        Queue::fake();
        // This didn't seem to work in setUpBeforeClass() for some reason
        self::$user = User::factory()->create();
        self::$customer = Customer::factory()->create();
        self::$shippingMethod = ShippingMethod::factory()->create();

        $dependentData = $this->assertTagsAreAddedOnCreate();
        $dependentData = $this->assertTagsAreChangedOnUpdate($dependentData);
        $dependentData = $this->assertTagsAreUnchangedWhenOmittedFromRequest($dependentData);
        $this->assertTagsAreRemovedWithEmptyArray($dependentData);
    }

    protected function assertTagsAreAddedOnCreate(): array
    {
        $tag1 = 'mytag1_'.rand().time();
        $tag2 = 'mytag2_'.rand().time();

        $payload = [
            'tags' => [
                $tag1,
                $tag2,
            ],
            'customer_id' => self::$customer->id,
            'shipping_address_id' => self::$customer->default_shipping_address_id,
            'billing_address_id' => self::$customer->default_billing_address_id,
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => '2022-10-19T00:07:41.260Z',
            'is_tax_included' => false,
            'tax_rate' => null,
            'tax_rate_id' => null,
            'store_name' => 'SKU.io',
            'store_id' => 1,
            'deliver_by_date' => '2022-10-25T00:07:41.221Z',
            'ship_by_date' => '2022-10-19T00:07:41.221Z',
            'shipping_method_id' => self::$shippingMethod->id,
            'discount_lines' => [
            ],
            'sales_order_lines' => [
                0 => [
                    'description' => 'test',
                    'quantity' => 1,
                    'discount' => 0,
                    'tax_rate' => null,
                    'tax_rate_id' => null,
                    'amount' => 0,
                ],
            ],
        ];

        $response = $this->actingAs(self::$user)
            ->postJson('/api/sales-orders', $payload);
        $response->assertOk();

        $salesOrderId = $response->json()['data']['id'];

        // Now fetch the order just created via the API
        // And see if the tags are included
        $response = $this->actingAs(self::$user)
            ->getJson('/api/sales-orders/'.$salesOrderId);
        $response->assertOk();

        $salesOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag1,
                $tag2,
            ],
            $salesOrder['tags']
        );

        return [$salesOrderId];
    }

    protected function assertTagsAreChangedOnUpdate(array $params): array
    {
        [$salesOrderId] = $params;

        $tag3 = 'mytag3_'.rand().time();

        $payload = [
            'tags' => [
                $tag3,
            ],
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags have been updated
        $response = $this->actingAs(self::$user)
            ->getJson('/api/sales-orders/'.$salesOrderId);
        $response->assertOk();

        $salesOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $salesOrder['tags']
        );

        return [$salesOrderId, $tag3];
    }

    protected function assertTagsAreUnchangedWhenOmittedFromRequest(array $params): array
    {
        [$salesOrderId, $tag3] = $params;

        $payload = [
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags are unchanged
        $response = $this->actingAs(self::$user)
            ->getJson('/api/sales-orders/'.$salesOrderId);
        $response->assertOk();

        $salesOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $salesOrder['tags']
        );

        $this->assertEquals(
            [
                $tag3,
            ],
            $salesOrder['tags']
        );

        return [$salesOrderId];
    }

    protected function assertTagsAreRemovedWithEmptyArray(array $params)
    {
        [$salesOrderId] = $params;

        $payload = [
            'tags' => [],
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/sales-orders/'.$salesOrderId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if all tags have been removed
        $response = $this->actingAs(self::$user)
            ->getJson('/api/sales-orders/'.$salesOrderId);
        $response->assertOk();

        $salesOrder = $response->json()['data'];

        $this->assertEquals(
            [],
            $salesOrder['tags']
        );
    }

    public function test_get_priority_warehouse_for_product(): void
    {
        $integrationInstance = SalesChannel::query()->find(1)->integrationInstance;
        IntegrationInstance::query()->find(1)->update(['integration_id' => 5, 'integration_settings' => ['auto_sync_enabled' => false]]);

        DataImportMapping::factory()->create([
            'integration_instance_id' => $integrationInstance->id,
            'model' => 'listings',
            'mapping' => json_decode('[{"listing_field":"sku","sku_field":"sku","parsers":[]},{"listing_field":"barcode","sku_field":"barcode","parsers":[]},{"listing_field":"variant_title","sku_field":"name","parsers":[]},{"listing_field":"price","sku_field":"price.Retail.value","parsers":[]},{"listing_field":"unit_cost","sku_field":"unit_cost","parsers":[]},{"listing_field":"vendor","sku_field":"default_supplier","parsers":[]},{"listing_field":"weight","sku_field":"weight","parsers":[]},{"listing_field":"weight_unit","sku_field":"weight_unit","parsers":[]},{"listing_field":"tags","sku_field":"tags","parsers":[]},{"listing_field":"image1","sku_field":"image","parsers":[]},{"listing_field":"image2","sku_field":"other_images","parsers":[]},{"listing_field":"image3","sku_field":"other_images","parsers":[]},{"listing_field":"image4","sku_field":"other_images","parsers":[]},{"listing_field":"image5","sku_field":"other_images","parsers":[]},{"listing_field":"image6","sku_field":"other_images","parsers":[]},{"listing_field":"image7","sku_field":"other_images","parsers":[]},{"listing_field":"image8","sku_field":"other_images","parsers":[]},{"listing_field":"image9","sku_field":"other_images","parsers":[]},{"listing_field":"image10","sku_field":"other_images","parsers":[]},{"listing_field":"image11","sku_field":"other_images","parsers":[]},{"listing_field":"image12","sku_field":"other_images","parsers":[]},{"listing_field":"image13","sku_field":"other_images","parsers":[]}]',
                1),
        ]);

        /** @var Warehouse $warehouseA */
        $warehouseA = Warehouse::factory()->create([
            'type' => 'direct',
            'name' => 'Los Angeles Warehouse',
            'address_id' => Address::factory()->create(['zip' => '90021'])->id,
        ]);

        /** @var Warehouse $warehouseB */
        $warehouseB = Warehouse::factory()->create([
            'type' => 'direct',
            'name' => 'Charlotte Warehouse',
            'address_id' => Address::factory()->create(['zip' => '28208'])->id,
        ]);

        Setting::query()->where('key', Setting::KEY_WAREHOUSE_PRIORITY)->update([
            'value' => json_encode("[$warehouseA->id, $warehouseB->id]"),
        ]);

        Setting::query()->where('key', Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER)->update([
            'value' => 0,
        ]);
        $productData = '{"id":44742653641013,"title":"Baby Bumble Onesie","body_html":"<meta charset=\"utf-8\">\n<p data-pm-slice=\"1 3 []\"><strong>Soft and comfortable onesie for newborns<\/strong><\/p>\n<ul>\n<li>\n<p>Fits Newborn<\/p>\n<\/li>\n<\/ul>","vendor":"Mamimoo Baby Clothing","product_type":"","created_at":"2023-03-26T22:07:43-04:00","handle":"baby-bumble-onesie","updated_at":"2023-05-09T08:00:14-04:00","published_at":"2023-03-26T22:07:43-04:00","template_suffix":"","status":"active","published_scope":"web","tags":"","admin_graphql_api_id":"gid:\/\/shopify\/ProductVariant\/44742653641013","options":[{"id":10406532120885,"product_id":8201861366069,"name":"Title","position":1,"values":["Default Title"]}],"images":[{"id":40951766483253,"product_id":8201861366069,"position":1,"created_at":"2023-03-26T22:07:44-04:00","updated_at":"2023-03-26T22:07:44-04:00","alt":null,"width":512,"height":512,"src":"https:\/\/cdn.shopify.com\/s\/files\/1\/0741\/7420\/2165\/products\/AHjexCj.jpg?v=1679882864","variant_ids":[],"admin_graphql_api_id":"gid:\/\/shopify\/ProductImage\/40951766483253"}],"image":"https:\/\/cdn.shopify.com\/s\/files\/1\/0741\/7420\/2165\/products\/AHjexCj.jpg?v=1679882864","product_id":8201861366069,"price":15.99,"sku":"BB001","position":1,"inventory_policy":"continue","compare_at_price":null,"fulfillment_service":"manual","inventory_management":"shopify","option1":"Default Title","option2":null,"option3":null,"taxable":false,"barcode":"123456789012","grams":454,"image_id":null,"weight":1,"weight_unit":"lb","inventory_item_id":46791574946101,"inventory_quantity":-1,"old_inventory_quantity":-1,"requires_shipping":true,"integration_instance_id":3,"image_count":1,"variant_title":"Baby Bumble Onesie","removed_from_shopify":0,"product_title":"Baby Bumble Onesie"}';
        $orderData = '{"id":5371992375605,"admin_graphql_api_id":"gid:\/\/shopify\/Order\/5371992375605","app_id":1354745,"browser_ip":"172.250.4.77","buyer_accepts_marketing":false,"cancel_reason":null,"cancelled_at":null,"cart_token":null,"checkout_id":36675835756853,"checkout_token":"811aa65df3b3afd3fb16812ef50b8834","client_details":{"accept_language":null,"browser_height":null,"browser_ip":"172.250.4.77","browser_width":null,"session_hash":null,"user_agent":"Mozilla\/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit\/537.36 (KHTML, like Gecko) Chrome\/112.0.0.0 Safari\/537.36"},"closed_at":null,"company":null,"confirmed":true,"contact_email":"rick@sku.io","created_at":"2023-05-09T07:59:35-04:00","currency":"USD","current_subtotal_price":"15.99","current_subtotal_price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"current_total_additional_fees_set":null,"current_total_discounts":"0.00","current_total_discounts_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"current_total_duties_set":null,"current_total_price":"15.99","current_total_price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"current_total_tax":"0.00","current_total_tax_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"customer_locale":"en","device_id":null,"discount_codes":[],"email":"rick@sku.io","estimated_taxes":false,"financial_status":"paid","fulfillment_status":null,"landing_site":null,"landing_site_ref":null,"location_id":null,"merchant_of_record_app_id":null,"name":"#1006","note":null,"note_attributes":[],"number":6,"order_number":1006,"order_status_url":"https:\/\/mamimoo-baby-clothing.myshopify.com\/74174202165\/orders\/b4bde109b93cc68dd2d5dc5f3d64f269\/authenticate?key=dc21ea391ba272119458b3f569370268","original_total_additional_fees_set":null,"original_total_duties_set":null,"payment_gateway_names":["manual"],"phone":null,"presentment_currency":"USD","processed_at":"2023-05-09T07:59:35-04:00","reference":"cb33a2a9b0e28e382594d18fa3ecf3e9","referring_site":null,"source_identifier":"cb33a2a9b0e28e382594d18fa3ecf3e9","source_name":"shopify_draft_order","source_url":null,"subtotal_price":"15.99","subtotal_price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"tags":"","tax_lines":[],"taxes_included":false,"test":false,"token":"b4bde109b93cc68dd2d5dc5f3d64f269","total_discounts":"0.00","total_discounts_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"total_line_items_price":"15.99","total_line_items_price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"total_outstanding":"0.00","total_price":"15.99","total_price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"total_shipping_price_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"total_tax":"0.00","total_tax_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"total_tip_received":"0.00","total_weight":453,"updated_at":"2023-05-09T07:59:36-04:00","user_id":94787109173,"billing_address":{"first_name":"Rick","address1":"500 Westover Drive","phone":"+17608808820","city":"Sanford","zip":"27330","province":"North Carolina","country":"United States","last_name":"Jacobs","address2":"#140-2671","company":null,"latitude":35.462585,"longitude":-79.2063632,"name":"Rick Jacobs","country_code":"US","province_code":"NC"},"customer":{"id":6939214414133,"email":"rick@sku.io","accepts_marketing":false,"created_at":"2023-04-17T12:09:54-04:00","updated_at":"2023-05-09T07:59:36-04:00","first_name":"Rick","last_name":"Jacobs","state":"disabled","note":null,"verified_email":true,"multipass_identifier":null,"tax_exempt":false,"phone":null,"email_marketing_consent":{"state":"not_subscribed","opt_in_level":"single_opt_in","consent_updated_at":null},"sms_marketing_consent":null,"tags":"","currency":"USD","accepts_marketing_updated_at":"2023-04-17T12:09:54-04:00","marketing_opt_in_level":null,"tax_exemptions":[],"admin_graphql_api_id":"gid:\/\/shopify\/Customer\/6939214414133","default_address":{"id":9181613392181,"customer_id":6939214414133,"first_name":"Rick","last_name":"Jacobs","company":"","address1":"500 Westover Drive","address2":"#140-2671","city":"Sanford","province":"North Carolina","country":"United States","zip":"27330","phone":"+17608808820","name":"Rick Jacobs","province_code":"NC","country_code":"US","country_name":"United States","default":true}},"discount_applications":[],"fulfillments":[],"line_items":[{"id":13939075973429,"admin_graphql_api_id":"gid:\/\/shopify\/LineItem\/13939075973429","fulfillable_quantity":1,"fulfillment_service":"manual","fulfillment_status":null,"gift_card":false,"grams":454,"name":"Baby Bumble Onesie","price":"15.99","price_set":{"shop_money":{"amount":"15.99","currency_code":"USD"},"presentment_money":{"amount":"15.99","currency_code":"USD"}},"product_exists":true,"product_id":8201861366069,"properties":[],"quantity":1,"requires_shipping":true,"sku":"BB001","taxable":false,"title":"Baby Bumble Onesie","total_discount":"0.00","total_discount_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"variant_id":44742653641013,"variant_inventory_management":"shopify","variant_title":null,"vendor":"Mamimoo Baby Clothing","tax_lines":[],"duties":[],"discount_allocations":[]}],"payment_terms":null,"refunds":[],"shipping_address":{"first_name":"Rick","address1":"500 Westover Drive","phone":"+17608808820","city":"Sanford","zip":"27330","province":"North Carolina","country":"United States","last_name":"Jacobs","address2":"#140-2671","company":null,"latitude":35.462585,"longitude":-79.2063632,"name":"Rick Jacobs","country_code":"US","province_code":"NC"},"shipping_lines":[]}';

        /** @var ShopifyProduct $shopifyProduct */
        $shopifyProduct = ShopifyProduct::factory()->create([
            'variant_id' => '44742653641013',
            'unit_cost' => 5,
            'integration_instance_id' => $integrationInstance->id,
            'json_object' => json_decode($productData, true),
        ]);

        $shopifyProduct->createSkuProduct([
            [
                'listing_field' => 'sku',
                'sku_field' => 'sku',
            ],
        ]);

        /** @var ShopifyOrder $shopifyOrder */
        $shopifyOrder = ShopifyOrder::factory()->create([
            'order_number' => '1006',
            'integration_instance_id' => $integrationInstance->id,
            'json_object' => json_decode($orderData, true),
        ]);

        dispatch_sync(new ShopifyCreateSkuProducts($integrationInstance, [$shopifyProduct->id]));

        $shopifyOrder->createSKUOrder();

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = SalesOrderLine::query()->first();

        $this->assertEquals($warehouseA->id, $salesOrderLine->warehouse_id);
        SalesOrder::query()->first()->delete();

        Setting::query()->where('key', Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER)->update([
            'value' => 1,
        ]);

        $shopifyOrder->fresh()->createSKUOrder();
        $this->assertEquals($warehouseB->id, SalesOrderLine::first()->warehouse_id);
    }

    public function test_it_can_auto_split_sales_order_line_across_warehouses(): void
    {
        $integrationInstance = SalesChannel::query()->find(1)->integrationInstance;
        IntegrationInstance::query()->find(1)->update(['integration_id' => 5]);

        DataImportMapping::factory()->create([
            'integration_instance_id' => $integrationInstance->id,
            'model' => 'listings',
            'mapping' => json_decode('[{"listing_field":"sku","sku_field":"sku","parsers":[]},{"listing_field":"barcode","sku_field":"barcode","parsers":[]},{"listing_field":"variant_title","sku_field":"name","parsers":[]},{"listing_field":"price","sku_field":"price.Retail.value","parsers":[]},{"listing_field":"unit_cost","sku_field":"unit_cost","parsers":[]},{"listing_field":"vendor","sku_field":"default_supplier","parsers":[]},{"listing_field":"weight","sku_field":"weight","parsers":[]},{"listing_field":"weight_unit","sku_field":"weight_unit","parsers":[]},{"listing_field":"tags","sku_field":"tags","parsers":[]},{"listing_field":"image1","sku_field":"image","parsers":[]},{"listing_field":"image2","sku_field":"other_images","parsers":[]},{"listing_field":"image3","sku_field":"other_images","parsers":[]},{"listing_field":"image4","sku_field":"other_images","parsers":[]},{"listing_field":"image5","sku_field":"other_images","parsers":[]},{"listing_field":"image6","sku_field":"other_images","parsers":[]},{"listing_field":"image7","sku_field":"other_images","parsers":[]},{"listing_field":"image8","sku_field":"other_images","parsers":[]},{"listing_field":"image9","sku_field":"other_images","parsers":[]},{"listing_field":"image10","sku_field":"other_images","parsers":[]},{"listing_field":"image11","sku_field":"other_images","parsers":[]},{"listing_field":"image12","sku_field":"other_images","parsers":[]},{"listing_field":"image13","sku_field":"other_images","parsers":[]}]',
                1),
        ]);
        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouseA */
        $warehouseA = Warehouse::factory()->create([
            'type' => 'direct',
            'name' => 'Los Angeles Warehouse',
            'address_id' => Address::factory()->create(['zip' => '90021'])->id,
        ])->withDefaultLocation();

        /** @var Warehouse $warehouseB */
        $warehouseB = Warehouse::factory()->create([
            'type' => 'direct',
            'name' => 'Charlotte Warehouse',
            'address_id' => Address::factory()->create(['zip' => '28208'])->id,
        ])->withDefaultLocation();

        Setting::query()->where('key', Setting::KEY_WAREHOUSE_PRIORITY)->update([
            'value' => json_encode("[$warehouseA->id, $warehouseB->id]"),
        ]);

        Setting::query()->where('key', Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER)->update([
            'value' => 0,
        ]);
        $productData = '{"id":40583548305517,"title":"iphone","body_html":"","vendor":"skuio","product_type":"","created_at":"2023-03-17T14:25:59-07:00","handle":"iphone","updated_at":"2023-03-17T14:26:00-07:00","published_at":"2023-03-17T14:25:59-07:00","template_suffix":"","status":"active","published_scope":"web","tags":"","admin_graphql_api_id":"gid:\/\/shopify\/ProductVariant\/40583548305517","options":[{"id":9001236430957,"product_id":6987690901613,"name":"Title","position":1,"values":["Default Title"]}],"images":[],"image":null,"product_id":6987690901613,"price":499,"sku":"iphone","position":1,"inventory_policy":"deny","compare_at_price":null,"fulfillment_service":"manual","inventory_management":"shopify","option1":"Default Title","option2":null,"option3":null,"taxable":true,"barcode":"","grams":2268,"image_id":null,"weight":5,"weight_unit":"lb","inventory_item_id":42682112737389,"inventory_quantity":0,"old_inventory_quantity":0,"requires_shipping":true,"integration_instance_id":10,"image_count":0,"variant_title":"iphone","removed_from_shopify":0,"product_title":"iphone"}';
        $orderData = '{"id":5547602051181,"admin_graphql_api_id":"gid:\/\/shopify\/Order\/5547602051181","app_id":1354745,"browser_ip":"45.130.96.165","buyer_accepts_marketing":false,"cancel_reason":null,"cancelled_at":null,"cart_token":null,"checkout_id":27886353186925,"checkout_token":"1720a708775c17ae9e435164eacb37ff","client_details":{"accept_language":null,"browser_height":null,"browser_ip":"45.130.96.165","browser_width":null,"session_hash":null,"user_agent":"Mozilla\/5.0 (X11; Ubuntu; Linux x86_64; rv:109.0) Gecko\/20100101 Firefox\/111.0"},"closed_at":null,"company":null,"confirmed":true,"contact_email":null,"created_at":"2023-05-28T02:49:58-07:00","currency":"USD","current_subtotal_price":"3992.00","current_subtotal_price_set":{"shop_money":{"amount":"3992.00","currency_code":"USD"},"presentment_money":{"amount":"3992.00","currency_code":"USD"}},"current_total_additional_fees_set":null,"current_total_discounts":"0.00","current_total_discounts_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"current_total_duties_set":null,"current_total_price":"4271.44","current_total_price_set":{"shop_money":{"amount":"4271.44","currency_code":"USD"},"presentment_money":{"amount":"4271.44","currency_code":"USD"}},"current_total_tax":"279.44","current_total_tax_set":{"shop_money":{"amount":"279.44","currency_code":"USD"},"presentment_money":{"amount":"279.44","currency_code":"USD"}},"customer_locale":"en","device_id":null,"discount_codes":[],"email":"","estimated_taxes":false,"financial_status":"paid","fulfillment_status":null,"landing_site":null,"landing_site_ref":null,"location_id":61761192045,"merchant_of_record_app_id":null,"name":"#1020","note":null,"note_attributes":[],"number":20,"order_number":1020,"order_status_url":"https:\/\/skuio.myshopify.com\/54898557037\/orders\/97e3130635314878aaaf20f79cbfa985\/authenticate?key=22e00726604793c1d5521bd906eb48ce","original_total_additional_fees_set":null,"original_total_duties_set":null,"payment_gateway_names":["manual"],"phone":null,"presentment_currency":"USD","processed_at":"2023-05-28T02:49:58-07:00","reference":"40d1738450079df5604938768d9110fe","referring_site":null,"source_identifier":"40d1738450079df5604938768d9110fe","source_name":"shopify_draft_order","source_url":null,"subtotal_price":"3992.00","subtotal_price_set":{"shop_money":{"amount":"3992.00","currency_code":"USD"},"presentment_money":{"amount":"3992.00","currency_code":"USD"}},"tags":"","tax_lines":[{"price":"189.62","rate":0.0475,"title":"North Carolina State Tax","price_set":{"shop_money":{"amount":"189.62","currency_code":"USD"},"presentment_money":{"amount":"189.62","currency_code":"USD"}},"channel_liable":false},{"price":"89.82","rate":0.0225,"title":"Lee County Tax","price_set":{"shop_money":{"amount":"89.82","currency_code":"USD"},"presentment_money":{"amount":"89.82","currency_code":"USD"}},"channel_liable":false}],"taxes_included":false,"test":false,"token":"97e3130635314878aaaf20f79cbfa985","total_discounts":"0.00","total_discounts_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"total_line_items_price":"3992.00","total_line_items_price_set":{"shop_money":{"amount":"3992.00","currency_code":"USD"},"presentment_money":{"amount":"3992.00","currency_code":"USD"}},"total_outstanding":"0.00","total_price":"4271.44","total_price_set":{"shop_money":{"amount":"4271.44","currency_code":"USD"},"presentment_money":{"amount":"4271.44","currency_code":"USD"}},"total_shipping_price_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"total_tax":"279.44","total_tax_set":{"shop_money":{"amount":"279.44","currency_code":"USD"},"presentment_money":{"amount":"279.44","currency_code":"USD"}},"total_tip_received":"0.00","total_weight":18136,"updated_at":"2023-05-28T02:50:05-07:00","user_id":73929621613,"billing_address":null,"customer":{"id":5708015239277,"email":null,"accepts_marketing":false,"created_at":"2022-05-29T07:13:50-07:00","updated_at":"2023-05-28T02:50:09-07:00","first_name":"Kalvin","last_name":"Mizzi","state":"disabled","note":null,"verified_email":true,"multipass_identifier":null,"tax_exempt":false,"phone":null,"email_marketing_consent":null,"sms_marketing_consent":null,"tags":"","currency":"USD","accepts_marketing_updated_at":"2023-05-28T02:50:09-07:00","marketing_opt_in_level":null,"tax_exemptions":[],"admin_graphql_api_id":"gid:\/\/shopify\/Customer\/5708015239277","default_address":{"id":6969227903085,"customer_id":5708015239277,"first_name":"Kalvin","last_name":"Mizzi","company":"","address1":"1455 4th Street","address2":"405","city":"Santa Monica","province":"Alabama","country":"United States","zip":"90401","phone":"+17608808820","name":"Kalvin Mizzi","province_code":"AL","country_code":"US","country_name":"United States","default":true}},"discount_applications":[],"fulfillments":[],"line_items":[{"id":12698429587565,"admin_graphql_api_id":"gid:\/\/shopify\/LineItem\/12698429587565","fulfillable_quantity":8,"fulfillment_service":"manual","fulfillment_status":null,"gift_card":false,"grams":2268,"name":"iphone","price":"499.00","price_set":{"shop_money":{"amount":"499.00","currency_code":"USD"},"presentment_money":{"amount":"499.00","currency_code":"USD"}},"product_exists":true,"product_id":6987690901613,"properties":[],"quantity":8,"requires_shipping":true,"sku":"iphone","taxable":true,"title":"iphone","total_discount":"0.00","total_discount_set":{"shop_money":{"amount":"0.00","currency_code":"USD"},"presentment_money":{"amount":"0.00","currency_code":"USD"}},"variant_id":40583548305517,"variant_inventory_management":"shopify","variant_title":null,"vendor":"skuio","tax_lines":[{"channel_liable":false,"price":"189.62","price_set":{"shop_money":{"amount":"189.62","currency_code":"USD"},"presentment_money":{"amount":"189.62","currency_code":"USD"}},"rate":0.0475,"title":"North Carolina State Tax"},{"channel_liable":false,"price":"89.82","price_set":{"shop_money":{"amount":"89.82","currency_code":"USD"},"presentment_money":{"amount":"89.82","currency_code":"USD"}},"rate":0.0225,"title":"Lee County Tax"}],"duties":[],"discount_allocations":[]}],"payment_terms":null,"refunds":[],"shipping_address":null,"shipping_lines":[]}';

        /** @var ShopifyProduct $shopifyProduct */
        $shopifyProduct = ShopifyProduct::factory()->create([
            'variant_id' => '40583548305517',
            'unit_cost' => 1000,
            'integration_instance_id' => $integrationInstance->id,
            'json_object' => json_decode($productData, true),
        ]);

        $shopifyProduct->createSkuProduct([
            [
                'listing_field' => 'sku',
                'sku_field' => 'sku',
            ],
        ]);

        /** @var ShopifyOrder $shopifyOrder */
        $shopifyOrder = ShopifyOrder::factory()->create([
            'order_number' => '1020',
            'integration_instance_id' => $integrationInstance->id,
            'json_object' => json_decode($orderData, true),
        ]);

        dispatch_sync(new ShopifyCreateSkuProducts($integrationInstance, [$shopifyProduct->id]));
        $timeOfOrder = Carbon::now()->format('Y-m-d H:i:s');
        $product = Product::first();
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouseA->id,
            'unit_cost' => 5.00,
            'quantity' => 5,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $timeOfOrder,
            'product_id' => $product->id,
            'warehouse_id' => $warehouseB->id,
            'unit_cost' => 5.00,
            'quantity' => 3,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertSuccessful();
        $shopifyOrder->createSKUOrder();
        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => $warehouseA->id,
            'quantity' => 5,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => $warehouseB->id,
            'quantity' => 3,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_deleting_sales_orders_deletes_sales_order_line_financials(): void
    {
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines()
            ->create();

        $salesOrderLine = SalesOrderLine::first();

        $this->assertDatabaseHas((new SalesOrderLineFinancial())->getTable(), [
            'sales_order_line_id' => $salesOrderLine->id,
        ]);

        app(SalesOrderRepository::class)->bulkDelete(Arr::wrap($salesOrder->id));

        $this->assertDatabaseEmpty((new SalesOrderLineFinancial())->getTable());
    }
}
