<?php

namespace Tests\Feature;

use App\Jobs\DeleteProductInventoryJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Customer;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Services\SalesOrder\LineReservationAction;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseOld;

class SalesOrderLineControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_reserve_inventory_for_selected_sales_order_lines(): void
    {
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                    'externally_fulfilled_quantity' => 0,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        // Un-process line item.
        $data = $response->json('data');
        $orderLineId = $data['item_info'][0]['sales_order_line_id'];

        $this->postJson('/api/sales-order-lines/reservations', [
            'sales_order_id' => $data['id'],
            'sales_order_lines' => [
                [
                    'id' => $orderLineId,
                    'quantity' => 5,
                    'action' => LineReservationAction::REVERSE_RESERVATION(),
                ],  // reverse all quantity
            ],
        ])->assertSuccessful();

        // Reserve inventory
        $this->postJson('/api/sales-order-lines/reservations', [
            'sales_order_id' => $data['id'],
            'sales_order_lines' => [
                [
                    'id' => $orderLineId,
                    'quantity' => 5,
                    'action' => LineReservationAction::RESERVE(),
                ],  // reverse all quantity
            ],
        ])->assertSuccessful();

        // Inventory movements should be created.
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);
    }

    public function test_it_cannot_reverse_more_quantity_than_reserved_for_selected_sales_order_lines(): void
    {
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                    'externally_fulfilled_quantity' => 0,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        // Inventory movements should be created.
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        // Un-process line item.
        $data = $response->json('data');
        $orderLineId = $data['item_info'][0]['sales_order_line_id'];

        $this->postJson('/api/sales-order-lines/reservations', [
            'sales_order_id' => $data['id'],
            'sales_order_lines' => [
                [
                    'id' => $orderLineId,
                    'quantity' => 10,
                    'action' => LineReservationAction::REVERSE_RESERVATION(),
                ],  // reverse all quantity
            ],
        ])->assertStatus(500);

        // Movements should still exist.
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);
    }

    public function test_it_can_reverse_reservation_for_selected_sales_order_lines(): void
    {
        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                    'externally_fulfilled_quantity' => 0,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        // Inventory movements should be created.
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        // Un-process line item.
        $data = $response->json('data');
        $orderLineId = $data['item_info'][0]['sales_order_line_id'];

        $this->postJson('/api/sales-order-lines/reservations', [
            'sales_order_id' => $data['id'],
            'sales_order_lines' => [
                [
                    'id' => $orderLineId,
                    'quantity' => 5,
                    'action' => LineReservationAction::REVERSE_RESERVATION(),
                ], // reverse all quantity
            ],
        ])->assertSuccessful();

        // There shouldn't be any sales related inventory movements
        $this->assertDatabaseMissing('inventory_movements', [
            'link_id' => $orderLineId,
            'link_type' => SalesOrderLine::class,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
    }

    public function test_it_can_convert_to_financial_line(): void
    {
        $this->postJson(route('sales-order-lines.convert-to-financial-line', [
            'salesOrderLine' => SalesOrderLine::factory()->create(),
            'financialLineType' => FinancialLineType::factory()->create(),
        ]))->assertSuccessful();

        $this->assertDatabaseCount((new FinancialLine())->getTable(), 1);
    }
}
