<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\DeleteProductInventoryJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Address;
use App\Models\BackorderQueue;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\Fluent\AssertableJson;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Tests\TestCaseOld;

class SalesOrderControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_increases_backordered_quantity_when_sales_order_line_quantity_increases()
    {

        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'amount' => 10,
                    'quantity' => 10,
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse->id,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 10,
            'released_quantity' => 0,
        ]);
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 10,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
        ]);

        // Create adjustment to release the backorder queue
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 4,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
        ])->assertSuccessful();

        // Backorder should be released
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 10,
            'released_quantity' => 4,
        ]);

        // Increase sol quantity so the backorder queue quantity can increase.
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 11,
                    'description' => $product->name,
                    'amount' => 10,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 11,
            'released_quantity' => 4,
        ]);

        $this->assertDatabaseCount('inventory_movements', 5);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -4,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => FifoLayer::class,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 4,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => FifoLayer::class,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 4,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => FifoLayer::class,
            'type' => InventoryMovement::TYPE_ADJUSTMENT,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -7,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
            'type' => InventoryMovement::TYPE_SALE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 7,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
            'type' => InventoryMovement::TYPE_SALE,
        ]);

    }

    public function test_it_wont_change_warehouse_for_lines_with_fulfillments(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        $product = Product::factory()->create();
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse1->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5,
            'quantity' => 10,
        ])->assertSuccessful();

        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse1->id,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse1->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);
        $this->assertDatabaseHas('fifo_layers', [
            'warehouse_id' => $warehouse1->id,
            'product_id' => $product->id,
        ]);

        // Fulfill
        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
            'fulfilled_at' => now(),
            'warehouse_id' => $warehouse1->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 5,
                ],
            ],
        ])->assertSuccessful();

        // Change warehouse
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse2->id,
                ],
            ],
        ])->assertStatus(400); // Shouldn't allow change of warehouse

        // Warehouse shouldn't change.
        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse1->id,
        ]);

        $this->assertDatabaseCount('inventory_movements', 4);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        // Line should be backordered
        $this->assertDatabaseCount('backorder_queues', 0);

        // Fifo warehouse shouldn't change
        $this->assertDatabaseHas('fifo_layers', [
            'warehouse_id' => $warehouse1->id,
            'product_id' => $product->id,
        ]);
    }

    public function test_it_can_change_warehouse_for_backordered_lines(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        $product = Product::factory()->create();

        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'sales_channel_line_id' => '123',
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse1->id,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse1->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 10,
            'released_quantity' => 0,
        ]);

        // Change warehouse
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'sales_channel_line_id' => '123',
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse2->id,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse2->id,
        ]);


        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse2->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_type' => BackorderQueue::class,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse2->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
            'layer_type' => BackorderQueue::class,
        ]);

        // Line should be backordered
        $this->assertDatabaseCount('backorder_queues', 1);

    }

    public function test_it_changes_movement_and_layer_warehouse_when_lines_warehouses_change(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        $product = Product::factory()->create();
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $warehouse1->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5,
            'quantity' => 10,
        ])->assertSuccessful();

        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse1->id,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse1->id,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);
        $this->assertDatabaseHas('fifo_layers', [
            'warehouse_id' => $warehouse1->id,
            'product_id' => $product->id,
        ]);

        // Change warehouse
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                    'warehouse_id' => $warehouse2->id,
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_lines', 1);
        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse2->id,
        ]);

        $this->assertDatabaseCount('inventory_movements', 3);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse2->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'warehouse_id' => $warehouse2->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
        ]);

        // Line should be backordered
        $this->assertDatabaseCount('backorder_queues', 1);

        // Fifo warehouse shouldn't change
        $this->assertDatabaseHas('fifo_layers', [
            'warehouse_id' => $warehouse1->id,
            'product_id' => $product->id,
        ]);
    }

    public function test_it_can_get_sales_order_details(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 2,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
                [
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();

        // Get sales order detail
        $response = $this->getJson('/api/sales-orders/'.$response->json('data.id'))->assertSuccessful();

        $response->assertJson(
            fn (AssertableJson $json) => $json->has('data.payments')
                ->has('data.purchase_orders')
                ->etc()

        );
    }

    public function test_it_update_inventory_movement_references_when_sales_order_number_changes(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ]
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST',
        ]);

        // Fulfill part of the lines.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $response->json('data.item_info.0.warehouse.id'),
            'quantity' => 2,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
            'fulfilled_at' => now(),
            'warehouse_id' => Warehouse::factory()->create()->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        // Update sales order number
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'sales_order_number' => 'SO-TEST-2',
        ])->assertSuccessful();

        $this->assertDatabaseHas('sales_orders', [
            'sales_order_number' => 'SO-TEST-2',
        ]);

        // Inventory movement references should be updated
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST-2',
            'link_type' => SalesOrderLine::class,
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST-2.1',
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);

        $this->assertDatabaseMissing('inventory_movements', [
            'reference' => 'SO-TEST',
        ]);
    }

    public function test_it_wont_change_inventory_movement_reference_when_sales_order_number_is_not_in_update(): void{

        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ]
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST',
        ]);

        // Fulfill part of the lines.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $response->json('data.item_info.0.warehouse.id'),
            'quantity' => 2,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
            'fulfilled_at' => now(),
            'warehouse_id' => Warehouse::factory()->create()->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        // Update sales order number
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'order_date' => now(),
        ])->assertSuccessful();

        // Inventory movement references should not be updated
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST',
            'link_type' => SalesOrderLine::class,
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST.1',
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);
    }


    public function test_it_wont_change_inventory_movement_reference_when_sales_order_number_value_is_not_changed(): void{

        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        /** @var Product $product */
        $product = Product::factory()->create();

        $payload = [
            'sales_order_number' => 'SO-TEST',
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => Customer::factory()->create()->id,
            'currency_code' => 'USD',
            'order_date' => now(),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'amount' => $this->faker->numberBetween(5, 10),
                    'description' => $this->faker->sentence(),
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ]
            ],
        ];

        // Perform request.
        $response = $this->postJson('/api/sales-orders', $payload)->assertSuccessful();
        $this->assertDatabaseCount('inventory_movements', 2);
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST',
        ]);

        // Fulfill part of the lines.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'warehouse_id' => $response->json('data.item_info.0.warehouse.id'),
            'quantity' => 2,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
            'fulfilled_at' => now(),
            'warehouse_id' => Warehouse::factory()->create()->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 2,
                ],
            ],
        ])->assertSuccessful();

        // Update sales order number
        $this->putJson('/api/sales-orders/'.$response->json('data.id'), [
            'order_date' => now(),
            'sales_order_number' => 'SO-TEST',
        ])->assertSuccessful();

        // Inventory movement references should not be updated
        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST',
            'link_type' => SalesOrderLine::class,
        ]);

        $this->assertDatabaseHas('inventory_movements', [
            'reference' => 'SO-TEST.1',
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);
    }


    public function test_it_can_choose_closest_warehouse_to_user_location(): void
    {
        Queue::fake([
            SyncBackorderQueueCoveragesJob::class,
            UpdateProductsInventoryAndAvgCost::class,
            DeleteProductInventoryJob::class,
        ]);

        Sanctum::actingAs(User::factory()->create());

        $address1 = Address::factory()->create(['zip' => '90210']);
        $address2 = Address::factory()->create(['zip' => '90211']);
        $address3 = Address::factory()->create(['zip' => '11225']);

        $warehouse2 = Warehouse::factory()->create(['address_id' => $address2->id])->withDefaultLocation();
        $warehouse3 = Warehouse::factory()->create(['address_id' => $address3->id])->withDefaultLocation();


        // Set settings values
        Setting::query()->upsert([
            [
                'key' => Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES,
                'value' => true,
            ],
            [
                'key' => Setting::KEY_WAREHOUSE_PRIORITY,
                'value' => json_encode([
                    $warehouse3->id, // Farthest warehouse has the highest priority.
                    $warehouse2->id, // 90211
                ]),
            ],
            [
                'key' => Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER,
                'value' => true,
            ],
        ], 'key');

        $product = Product::factory()->create();

        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'shipping_address_id' => $address1->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 10,
                    'amount' => 5,
                    'description' => $product->name,
                    'warehouse_routing_method' => WarehouseRoutingMethod::ADVANCED->value,
                ],
            ],
        ])->assertSuccessful();

        // 90211 warehouse should be selected as it's the closest to 90210
        $this->assertDatabaseHas('sales_order_lines', [
            'warehouse_id' => $warehouse2->id,
        ]);
    }
}
