<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Customer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Store;
use App\Models\User;
use App\Models\Warehouse;
use App\Notifications\SalesOrderFulfilledNotification;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Queue;
use Illuminate\Testing\Fluent\AssertableJson;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class SalesChannelEmailTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_wont_email_amazon_customers_about_order_fulfillments()
    {

        Queue::fake();
        Notification::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var IntegrationInstance $instance */
        $instance = IntegrationInstance::query()->firstOrCreate(
            ['name' => Integration::NAME_AMAZON_US],
            [
                'integration_id' => Integration::query()->firstOrCreate(
                    ['name' => Integration::NAME_AMAZON_US],
                )->id,
                'name' => Integration::NAME_AMAZON_US,
                'integration_settings' => [
                    'emailCustomers' => true,
                    'inventory' => []
                ],
                'connection_settings' => []
            ]
        );
        $salesChannel = SalesChannel::factory()->create(['integration_instance_id' => $instance->id]);

        $warehouse = Warehouse::factory()->create()->withDefaultLocation();
        $customer = Customer::factory()->create(['email' => $this->faker->email()]);

        // Create product with stock
        $product = Product::factory()->create();
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'quantity' => 3,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5,
        ])->assertSuccessful();

        // Create sales order
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now()->subMinutes(5),
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'sales_channel_id' => $salesChannel->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'amount' => 5,
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouse->id,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                ],
            ],
        ])->assertSuccessful();

        // Fulfill Sales Order
        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfilled_at' => now(),
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 3,
                ],
            ],
        ])->assertSuccessful();

        // Assert that fulfillment email is sent to customer.
        Notification::assertNothingSentTo($customer);

    }

    public function test_it_emails_customers_about_fulfilled_orders_when_sales_channel_allows_customer_emails()
    {

        Queue::fake();
        Notification::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var IntegrationInstance $instance */
        $instance = IntegrationInstance::query()->firstOrCreate(
            ['name' => Integration::NAME_SHOPIFY],
            [
                'integration_id' => Integration::query()->firstOrCreate(
                    ['name' => Integration::NAME_SHOPIFY],
                )->id,
                'name' => Integration::NAME_SHOPIFY,
                'integration_settings' => [
                    'emailCustomers' => true,
                    'inventory' => []
                ],
                'connection_settings' => []
            ]
        );
        $salesChannel = SalesChannel::factory()->create(['integration_instance_id' => $instance->id]);

        $warehouse = Warehouse::factory()->create()->withDefaultLocation();
        $customer = Customer::factory()->create(['email' => $this->faker->email()]);

        // Create product with stock
        $product = Product::factory()->create();
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'quantity' => 3,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 5,
        ])->assertSuccessful();

        // Create sales order
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now()->subMinutes(5),
            'order_status' => SalesOrder::STATUS_OPEN,
            'customer_id' => $customer->id,
            'sales_channel_id' => $salesChannel->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'amount' => 5,
                    'description' => $this->faker->sentence(),
                    'warehouse_id' => $warehouse->id,
                    'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
                ],
            ],
        ])->assertSuccessful();

        // Fulfill Sales Order
        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfilled_at' => now(),
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 3,
                ],
            ],
        ])->assertSuccessful();

        // Assert that fulfillment email is sent to customer.
        Notification::assertSentTo(
            $customer,
            SalesOrderFulfilledNotification::class
        );

    }

    public function test_email_customer_setting_cannot_be_changed_for_amazon()
    {

        Queue::fake();
        Sanctum::actingAs(User::factory()->create(['is_power_user' => true]));

        /** @var IntegrationInstance $instance */
        $instance = IntegrationInstance::query()->firstOrCreate(
            ['name' => Integration::NAME_AMAZON_US],
            [
                'integration_id' => Integration::query()->firstOrCreate(
                    ['name' => Integration::NAME_AMAZON_US],
                )->id,
                'name' => Integration::NAME_AMAZON_US,
                'connection_settings' => [],
                'integration_settings' => [
                    'inventory' => [],
                ]
            ]
        );
        SalesChannel::factory()->create(['integration_instance_id' => $instance->id]);

        $response = $this->putJson('/api/integration-instances/'.$instance->id, [
            'integration_id' => Integration::factory()->create()->id,
            'name' => Integration::NAME_AMAZON_US,
            'integration_settings' => [
                'store' => ['id' => Store::factory()->create()->id],
                'emailCustomers' => true, // This should be reset.
                'inventory' => [],
            ],
            'connection_settings' => [],
        ])->assertSuccessful();

        $this->assertFalse(
            $response->json('data.integration_settings.emailCustomers')
        );

    }

    public function test_it_sets_default_customer_email_setting_for_integration_instance()
    {

        Queue::fake();

        Sanctum::actingAs(User::factory()->create(['is_power_user' => true]));

        $response = $this->postJson('/api/integration-instances', [
            'integration_id' => Integration::factory()->create()->id,
            'name' => Integration::NAME_SHOPIFY,
            'integration_settings' => [
                'settings' => [
                    'store' => ['id' => Store::factory()->create()->id],
                ],
                'inventory' => []
            ],
            'connection_settings' => [],
        ])->assertSuccessful();

        $response->assertJson(function (AssertableJson $json) {
            return $json->has('data.integration_settings.emailCustomers')->etc();
        });

    }
}
