<?php

namespace Tests\Feature;

use App\Models\Customer;
use App\Models\ShippingMethod;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

/**
 * @group testnow2
 */
class PurchaseOrderTest extends TestCase
{
    use FastRefreshDatabase;

    protected static $user;

    protected static $customer;

    protected static $shippingMethod;

    protected static $supplier;

    protected static $store;

    protected static $warehouse;

    public function test_tags(): void
    {
        // This didn't seem to work in setUpBeforeClass() for some reason
        self::$user = User::factory()->create();
        self::$customer = Customer::factory()->create();
        self::$shippingMethod = ShippingMethod::factory()->create();
        self::$warehouse = Warehouse::factory()->create();
        self::$supplier = Supplier::factory()->withWarehouse()->create();
        self::$supplier->default_warehouse_id = self::$warehouse->id;
        self::$supplier->save();

        self::$store = Store::factory()->create();

        $dependentData = $this->assertTagsAreAddedOnCreate();
        $dependentData = $this->assertTagsAreChangedOnUpdate($dependentData);
        $this->assertTagsAreUnchangedWhenOmittedFromRequest($dependentData);
    }

    protected function assertTagsAreAddedOnCreate()
    {
        $tag1 = 'mytag1_'.rand().time();
        $tag2 = 'mytag2_'.rand().time();

        $payload = [
            'is_tax_included' => false,
            'tags' => [
                $tag1,
                $tag2,
            ],
            'supplier_id' => self::$supplier->id,
            'purchase_order_date' => '2022-10-18T15:17:09.353Z',
            'purchase_order_lines' => [
                0 => [
                    'description' => 'test',
                    'quantity' => 1,
                    'discount' => 0,
                    'amount' => 0,
                    'is_product' => false,
                    'nominal_code_id' => 4,
                ],
            ],
            'store_id' => self::$store->id,
            'currency_code' => 'USD',
            'requested_shipping_method_id' => 8,
        ];

        $response = $this->actingAs(self::$user)
            ->postJson('/api/purchase-orders', $payload);
        $response->assertOk();

        $purchaseOrderId = $response->json()['data']['id'];

        // Now fetch the order just created via the API
        // And see if the tags are included
        $response = $this->actingAs(self::$user)
            ->getJson('/api/purchase-orders/'.$purchaseOrderId);
        $response->assertOk();

        $purchaseOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag1,
                $tag2,
            ],
            $purchaseOrder['tags']
        );

        return [$purchaseOrderId];
    }

    protected function assertTagsAreChangedOnUpdate(array $params): array
    {
        [$purchaseOrderId] = $params;

        $tag3 = 'mytag3_'.rand().time();

        $payload = [
            'tags' => [
                $tag3,
            ],
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/purchase-orders/'.$purchaseOrderId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags have been updated
        $response = $this->actingAs(self::$user)
            ->getJson('/api/purchase-orders/'.$purchaseOrderId);
        $response->assertOk();

        $purchaseOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $purchaseOrder['tags']
        );

        return [$purchaseOrderId, $tag3];
    }

    protected function assertTagsAreUnchangedWhenOmittedFromRequest(array $params): array
    {
        [$purchaseOrderId, $tag3] = $params;

        $payload = [
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/purchase-orders/'.$purchaseOrderId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags are unchanged
        $response = $this->actingAs(self::$user)
            ->getJson('/api/purchase-orders/'.$purchaseOrderId);
        $response->assertOk();

        $purchaseOrder = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $purchaseOrder['tags']
        );

        $this->assertEquals(
            [
                $tag3,
            ],
            $purchaseOrder['tags']
        );

        return [$purchaseOrderId];
    }
}
