<?php

namespace Tests\Feature;

use App\Models\Customer;
use App\Models\Product;
use App\Models\ShippingMethod;
use App\Models\User;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ProductTest extends TestCase
{
    use FastRefreshDatabase;

    protected static $user;

    protected static $customer;

    protected static $shippingMethod;

    public function test_tags(): void
    {
        Queue::fake();

        // This didn't seem to work in setUpBeforeClass() for some reason
        self::$user = User::factory()->create();
        self::$customer = Customer::factory()->create();
        self::$shippingMethod = ShippingMethod::factory()->create();

        $dependentData = $this->assertTagsAreAddedOnCreate();
        $dependentData = $this->assertTagsAreChangedOnUpdate($dependentData);
        $dependentData = $this->assertTagsAreUnchangedWhenOmittedFromRequest($dependentData);
        $this->assertTagsAreRemovedWithEmptyArray($dependentData);
    }

    protected function assertTagsAreAddedOnCreate()
    {
        $tag1 = 'mytag1_'.rand().time();
        $tag2 = 'mytag2_'.rand().time();

        $payload = [
            'name' => 'My Name',
            'sku' => 'MYSKU'.rand().time(),
            'type' => 'standard',
            'tags' => [
                $tag1,
                $tag2,
            ],
        ];

        $response = $this->actingAs(self::$user)
            ->postJson('/api/products', $payload);
        $response->assertOk();

        $productId = $response->json()['data']['id'];

        // Now fetch the order just created via the API
        // And see if the tags are included
        $response = $this->actingAs(self::$user)
            ->getJson('/api/products/'.$productId);
        $response->assertOk();

        $product = $response->json()['data'];

        $this->assertEquals(
            [
                $tag1,
                $tag2,
            ],
            $product['tags']
        );

        return [$productId];
    }

    protected function assertTagsAreChangedOnUpdate(array $params): array
    {
        [$productId] = $params;

        $tag3 = 'mytag3_'.rand().time();

        $payload = [
            'tags' => [
                $tag3,
            ],
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/products/'.$productId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags have been updated
        $response = $this->actingAs(self::$user)
            ->getJson('/api/products/'.$productId);
        $response->assertOk();

        $product = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $product['tags']
        );

        return [$productId, $tag3];
    }

    protected function assertTagsAreUnchangedWhenOmittedFromRequest(array $params): array
    {
        [$productId, $tag3] = $params;

        $payload = [
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/products/'.$productId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if the tags are unchanged
        $response = $this->actingAs(self::$user)
            ->getJson('/api/products/'.$productId);
        $response->assertOk();

        $product = $response->json()['data'];

        $this->assertEquals(
            [
                $tag3,
            ],
            $product['tags']
        );

        $this->assertEquals(
            [
                $tag3,
            ],
            $product['tags']
        );

        return [$productId];
    }

    protected function assertTagsAreRemovedWithEmptyArray(array $params)
    {
        [$productId] = $params;

        $payload = [
            'tags' => [],
        ];

        $response = $this->actingAs(self::$user)
            ->putJson('/api/products/'.$productId, $payload);
        $response->assertOk();

        // Now fetch the order
        // And see if all tags have been removed
        $response = $this->actingAs(self::$user)
            ->getJson('/api/products/'.$productId);
        $response->assertOk();

        $product = $response->json()['data'];

        $this->assertEquals(
            [],
            $product['tags']
        );
    }

    public function test_search_product_by_sku(): void
    {
        Sanctum::actingAs(User::factory()->create());

        $productsData = [
            [
                'sku' => 'GSL392-blemished-test blemish2',
                'name' => 'Walbro GSL392 255 LPH High Pressure Inline Fuel Pump (NO KIT INCLUDED)',
                'type' => Product::TYPE_STANDARD,
            ],
            [
                'sku' => 'GSL392+128-3039_2',
                'name' => 'Walbro GSL392 255 LPH High Pressure Inline Fuel Pump with 2x AN-6 Anodized Fittings',
                'type' => Product::TYPE_STANDARD,
            ],
            [
                'sku' => 'GSL392+2x128-3012',
                'name' => 'Walbro Universal 255 LPH Inline High Pressure Fuel Pump External GSL392 with 128-3012 Bundy Fittings',
                'type' => Product::TYPE_STANDARD,
            ],
            [
                'sku' => 'GSL392+128-3040_2',
                'name' => 'Walbro GSL392 255 LPH High Pressure Inline Fuel Pump with 2x AN-8 Anodized Fittings',
                'type' => Product::TYPE_STANDARD,
            ],
            [
                'sku' => 'GSL392+128-3024_2',
                'name' => 'Walbro GSL392 255 LPH High Pressure Inline Fuel Pump with 2x 8mm Single Barb Fittings',
                'type' => Product::TYPE_STANDARD,
            ],
            [
                'sku' => 'GSL392',
                'name' => 'Walbro GSL392 255 LPH High Pressure Inline Fuel Pump (NO KIT INCLUDED)',
                'type' => Product::TYPE_STANDARD,
            ],
        ];

        foreach ($productsData as $productData) {
            Product::factory()->create($productData);
        }

        $response = $this->getJson('/api/products?limit=10&filters={"conjunction":"and","filterSet":[]}&sortObjs=[]&page=1&query=GSL392&table_specifications=0&total=0&archived=0&visible_only=0&included=["sku","image","name","brand","price","average_cost","default_supplier","supplier_pricing","inventory_stock_value"]');
        $response->assertOk();
        $data = json_decode($response->getContent(), true)['data'];
        $dataSku = collect($data)->pluck('sku');
        $this->assertEquals('GSL392', $dataSku[0]);
    }
}
