<?php

namespace Tests\Feature;

use App\Data\IntegrationInstanceInventoryData;
use App\Exceptions\ProductBundleException;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Warehouse;
use Exception;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Managers\EbayProductManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ProductListingQuantityCacheUpdateTest extends TestCase
{

    use FastRefreshDatabase;

    /**
     * @throws Exception
     */
    public function test_it_can_update_listing_quantity_for_product(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel();

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 10,
        ]);
    }


    /**
     * @throws Exception
     */
    public function test_it_can_update_listing_quantity_only_for_warehouses_listed_in_inventory_rules(): void
    {
        /** @var Warehouse $anotherWarehouse */
        $anotherWarehouse = Warehouse::factory()->create()->withDefaultLocation();

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => 'None',
                    ],
                    'selectedWarehouses' => [$anotherWarehouse->id]
                ])
            );

        // Product inventory for the warehouse not listed in the inventory rules
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product inventory for the warehouse listed in the inventory rules
        $product->productInventory()->create([
            'warehouse_id' => $anotherWarehouse->id,
            'inventory_available' => 20,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated only for the warehouse listed in the inventory rules
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 20,
        ]);
    }


    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_fixed_allocation_minimum_inventory_rule(): void
    {
        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => 'Fixed Allocation',
                        'minRuleTypeValue' => 15,
                    ],
                ])
            );

        // Product inventory less than min value
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache set to the minimum value.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 15,
        ]);
    }


    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_percent_available_rule_for_minimum_quantity(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => '% of Available',
                        'minRuleTypeValue' => 50,
                        'subtractBufferStock' => 8
                    ],
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache set to 50% of what's available.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);
    }

    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_fixed_allocation_maximum_quantity(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'Fixed Allocation',
                        'maxRuleTypeValue' => 15,
                        'minRuleType' => 'None'
                    ],
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 20,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache set to the maximum value.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 15,
        ]);
    }


    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_percent_available_rule_for_maximum_quantity(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => '% of Available',
                        'maxRuleTypeValue' => 20,
                        'minRuleType' => 'None',
                        'subtractBufferStock' => 8
                    ],
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 20,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache set to 20% of what's available.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 4,
        ]);
    }


    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_buffer_stock_allowance(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => 'None',
                        'substractBufferStock' => 5,
                    ],
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated with buffer stock allowance.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);
    }


    /**
     * @return void
     * @throws Exception
     */
    public function test_it_can_apply_buffer_stock_and_maximum_quantity(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'Fixed Allocation',
                        'maxRuleTypeValue' => 10,
                        'minRuleType' => 'None',
                        'substractBufferStock' => 5,
                    ],
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 15,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated with buffer stock allowance and minimum quantity.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_apply_rules_when_override_exists_in_product_listing(): void{


        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel();

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 15,
        ]);

        // Product listing with rules
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
            'inventory_rules' => [
                'inventoryModificationRules' => [
                    'maxRuleType' => 'Fixed Allocation',
                    'maxRuleTypeValue' => 5,
                ],
            ],
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated with the rule overrides.
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);

    }


    /**
     * @throws Exception
     */
    public function test_it_can_update_listings_for_multiple_products(): void{

            [$product1, $warehouse, $salesChannel, $integrationInstance] =
                $this->createProductWithWarehouseAndSalesChannel();

            [$product2] =
                $this->createProductWithWarehouseAndSalesChannel(
                    integrationInstance: $integrationInstance,
                    warehouse: $warehouse
                );

            // Product 1 inventory
            $product1->productInventory()->create([
                'warehouse_id' => $warehouse->id,
                'inventory_available' => 15,
            ]);

            // Product 1 listing
            $product1->productListings()->create([
                'sales_channel_id' => $salesChannel->id,
                'quantity' => null,
                'sales_channel_listing_id' => '1111111111',
            ]);

            // Product 2 inventory
            $product2->productInventory()->create([
                'warehouse_id' => $warehouse->id,
                'inventory_available' => 20,
            ]);

            // Product 2 listing
            $product2->productListings()->create([
                'sales_channel_id' => $salesChannel->id,
                'quantity' => null,
                'sales_channel_listing_id' => '2222222222',
            ]);

            // Update the listing quantity cache
            /** @var EbayIntegrationInstance $ebayInstance */
            $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
            $manager = new EbayProductManager(
                ebayIntegrationInstance: $ebayInstance
            );
            $manager->cacheProductListingQuantity();

            // Assert that the listing quantity cache is updated for both products
            $this->assertDatabaseHas('product_listings', [
                'product_id' => $product1->id,
                'sales_channel_id' => $salesChannel->id,
                'quantity' => 15,
            ]);

            $this->assertDatabaseHas('product_listings', [
                'product_id' => $product2->id,
                'sales_channel_id' => $salesChannel->id,
                'quantity' => 20,
            ]);
    }


    /**
     * @throws Exception
     */
    public function test_it_can_update_listings_for_multiple_products_with_inventory_rules(): void{

        [$product1, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'Fixed Allocation',
                        'maxRuleTypeValue' => 10,
                        'minRuleType' => 'None',
                    ],
                ])
            );

        [$product2] =
            $this->createProductWithWarehouseAndSalesChannel(
                integrationInstance: $integrationInstance,
                warehouse: $warehouse
            );

        // Product 1 inventory
        $product1->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 15,
        ]);

        // Product 1 listing
        $product1->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Product 2 inventory
        $product2->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 20,
        ]);

        // Product 2 listing
        $product2->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '2222222222',
            'inventory_rules' => [
                'inventoryModificationRules' => [
                    'maxRuleType' => 'Fixed Allocation',
                    'maxRuleTypeValue' => 5,
                ],
            ],
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated for both products
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product1->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 10,
        ]);

        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product2->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);
    }

    /**
     * @throws ProductBundleException
     * @throws Exception
     */
    public function test_it_can_update_quantity_for_bundled_products(): void{

        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel();

        /** @var Product $bundledProduct */
        $bundledProduct = Product::factory()->create(
            ['type' => Product::TYPE_BUNDLE]
        );

        $bundledProduct->setBundleComponents([
            [
                'id' => $product->id,
                'quantity' => 2,
            ]
        ]);

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        $bundledProduct->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '2222222222',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Assert that the listing quantity cache is updated for the bundled product
        $this->assertDatabaseHas('product_listings', [
            'product_id' => $bundledProduct->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 5,
        ]);

        $this->assertDatabaseHas('product_listings', [
            'product_id' => $product->id,
            'sales_channel_id' => $salesChannel->id,
            'quantity' => 10,
        ]);
    }


    public function test_it_updates_quantity_for_inventory_locations(): void{


        [$product, $warehouse, $salesChannel, $integrationInstance] =
            $this->createProductWithWarehouseAndSalesChannel(
                inventoryRules: IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => 'None',
                    ],
                    'locations' => [
                        [
                            'id' => '1111111111',
                            'name' => 'Location 1',
                            'masterOfStock' => ProductListing::MASTER_SKU
                        ],
                        [
                            'id' => '2222222222',
                            'name' => 'Location 2',
                            'masterOfStock' => ProductListing::MASTER_SKU
                        ]
                    ]
                ])
            );

        // Product inventory
        $product->productInventory()->create([
            'warehouse_id' => $warehouse->id,
            'inventory_available' => 10,
        ]);

        // Product listing
        $product->productListings()->create([
            'sales_channel_id' => $salesChannel->id,
            'quantity' => null,
            'sales_channel_listing_id' => '1111111111',
        ]);

        // Update the listing quantity cache
        /** @var EbayIntegrationInstance $ebayInstance */
        $ebayInstance = EbayIntegrationInstance::query()->findOrFail($integrationInstance->id);
        $manager = new EbayProductManager(
            ebayIntegrationInstance: $ebayInstance
        );
        $manager->cacheProductListingQuantity();

        // Listing locations should exist.
        $this->assertDatabaseCount('product_listing_inventory_locations', 2);
        $this->assertDatabaseHas('product_listing_inventory_locations', [
            'sales_channel_location_id' => '1111111111',
            'quantity' => 10,
        ]);
        $this->assertDatabaseHas('product_listing_inventory_locations', [
            'sales_channel_location_id' => '2222222222',
            'quantity' => 10,
        ]);
    }


    /**
     * @param  IntegrationInstanceInventoryData|null  $inventoryRules
     * @param  IntegrationInstance|null  $integrationInstance
     * @param  Warehouse|null  $warehouse
     * @return array
     */
    private function createProductWithWarehouseAndSalesChannel(
        ?IntegrationInstanceInventoryData $inventoryRules = null,
        IntegrationInstance $integrationInstance = null,
        Warehouse $warehouse = null
    ): array
    {
        /** @var Product $product */
        $product = Product::factory()->create();

        if (empty($warehouse)) {
            $warehouse = Warehouse::factory()->create()->withDefaultLocation();
        }

        if (empty($integrationInstance)) {
            $integration = Integration::query()->where('name', Integration::NAME_EBAY)->firstOrFail();
            /** @var IntegrationInstance|EbayIntegrationInstance $integrationInstance */
            $integrationInstance = IntegrationInstance::factory()->create([
                'integration_id' => $integration->id,
            ]);

            $inventoryRules = !empty($inventoryRules) ? $inventoryRules :
                IntegrationInstanceInventoryData::from([
                    'inventoryModificationRules' => [
                        'maxRuleType' => 'None',
                        'minRuleType' => 'None',
                    ],
                    'selectedWarehouses' => [$warehouse->id],
                ]);

            $settings = $integrationInstance->integration_settings;
            $settings['inventory'] = $inventoryRules;
            $integrationInstance->update([
                'integration_settings' => $settings,
            ]);

            /** @var SalesChannel $salesChannel */
            $salesChannel = SalesChannel::factory()->create([
                'integration_instance_id' => $integrationInstance->id,
            ]);
        } else {
            $salesChannel = $integrationInstance->salesChannel;
        }

        return [$product, $warehouse, $salesChannel, $integrationInstance];
    }
}
